#!/bin/bash
# Qlik Cloud App Fields
# Get available fields in an app
# Usage: qlik-app-fields.sh <app-id>

set -euo pipefail

APP_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$APP_ID" ]]; then
  echo "{\"success\":false,\"error\":\"App ID required. Usage: qlik-app-fields.sh <app-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

# Get app metadata which includes table/field info
curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/apps/${APP_ID}/data/metadata" | python3 -c "
import json
import sys

app_id = '$APP_ID'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    # Extract fields (filter out system fields starting with \$)
    all_fields = data.get('fields', [])
    fields = []
    for f in all_fields:
        if not f.get('is_system', False) and not f.get('is_hidden', False):
            fields.append({
                'name': f.get('name'),
                'tables': f.get('src_tables', []),
                'cardinal': f.get('cardinal'),
                'isNumeric': f.get('is_numeric', False),
                'tags': f.get('tags', [])
            })
    
    # Extract tables
    tables = data.get('tables', [])
    table_list = [{'name': t.get('name'), 'rows': t.get('noOfRows')} for t in tables]
    
    print(json.dumps({
        'success': True,
        'appId': app_id,
        'fields': fields,
        'tables': table_list,
        'fieldCount': len(fields),
        'tableCount': len(tables),
        'timestamp': timestamp
    }, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
