#!/bin/bash
# Qlik Cloud Create App
# Create a new app
# Usage: qlik-app-create.sh "name" [space-id] [description]

set -euo pipefail

APP_NAME="${1:-}"
SPACE_ID="${2:-}"
DESCRIPTION="${3:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$APP_NAME" ]]; then
  echo "{\"success\":false,\"error\":\"App name required. Usage: qlik-app-create.sh \\\"name\\\" [space-id] [description]\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

# Build request body
BODY="{\"attributes\":{\"name\":$(echo "$APP_NAME" | python3 -c 'import sys,json; print(json.dumps(sys.stdin.read().strip()))')"
if [[ -n "$DESCRIPTION" ]]; then
  BODY="$BODY,\"description\":$(echo "$DESCRIPTION" | python3 -c 'import sys,json; print(json.dumps(sys.stdin.read().strip()))')"
fi
if [[ -n "$SPACE_ID" ]]; then
  BODY="$BODY,\"spaceId\":\"$SPACE_ID\""
fi
BODY="$BODY}}"

curl -sL -X POST \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  -d "$BODY" \
  "${TENANT}/api/v1/apps" | python3 -c "
import json
import sys

timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    attrs = data.get('attributes', data)
    result = {
        'success': True,
        'app': {
            'id': data.get('id'),
            'name': attrs.get('name'),
            'spaceId': attrs.get('spaceId'),
            'created': attrs.get('createdDate')
        },
        'message': 'App created successfully',
        'timestamp': timestamp
    }
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
