#!/bin/bash
# Qlik Cloud Trigger Alert
# Manually trigger a data alert evaluation
# Usage: qlik-alert-trigger.sh <alert-id>

set -euo pipefail

ALERT_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$ALERT_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Alert ID required. Usage: qlik-alert-trigger.sh <alert-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL -X POST \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/data-alerts/${ALERT_ID}/actions/evaluate" | python3 -c "
import json
import sys

alert_id = '$ALERT_ID'
timestamp = '$TIMESTAMP'

try:
    raw = sys.stdin.read().strip()
    
    if not raw:
        print(json.dumps({'success': True, 'alertId': alert_id, 'message': 'Alert evaluation triggered', 'timestamp': timestamp}, indent=2))
        sys.exit(0)
    
    data = json.loads(raw)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', data['errors'][0].get('detail', 'Unknown error')), 'alertId': alert_id, 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    print(json.dumps({'success': True, 'alertId': alert_id, 'result': data, 'timestamp': timestamp}, indent=2))
except json.JSONDecodeError:
    print(json.dumps({'success': True, 'alertId': alert_id, 'message': 'Alert evaluation triggered', 'timestamp': timestamp}, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
