#!/bin/bash
# Qlik Cloud Get Alert
# Get details of a specific data alert
# Usage: qlik-alert-get.sh <alert-id>

set -euo pipefail

ALERT_ID="${1:-}"
TIMESTAMP=$(date -u +%Y-%m-%dT%H:%M:%SZ)

if [[ -z "${QLIK_TENANT:-}" ]] || [[ -z "${QLIK_API_KEY:-}" ]]; then
  echo "{\"success\":false,\"error\":\"QLIK_TENANT and QLIK_API_KEY required\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

if [[ -z "$ALERT_ID" ]]; then
  echo "{\"success\":false,\"error\":\"Alert ID required. Usage: qlik-alert-get.sh <alert-id>\",\"timestamp\":\"$TIMESTAMP\"}"
  exit 1
fi

TENANT="${QLIK_TENANT%/}"
[[ "$TENANT" != http* ]] && TENANT="https://$TENANT"

curl -sL \
  -H "Authorization: Bearer ${QLIK_API_KEY}" \
  -H "Content-Type: application/json" \
  "${TENANT}/api/v1/data-alerts/${ALERT_ID}" | python3 -c "
import json
import sys

alert_id = '$ALERT_ID'
timestamp = '$TIMESTAMP'

try:
    data = json.load(sys.stdin)
    
    if 'errors' in data:
        print(json.dumps({'success': False, 'error': data['errors'][0].get('title', 'Unknown error'), 'alertId': alert_id, 'timestamp': timestamp}, indent=2))
        sys.exit(1)
    
    result = {
        'success': True,
        'alert': {
            'id': data.get('id'),
            'name': data.get('name'),
            'description': data.get('description'),
            'enabled': data.get('enabled'),
            'appId': data.get('appId'),
            'ownerId': data.get('ownerId'),
            'condition': data.get('condition'),
            'recipients': data.get('recipients'),
            'lastTriggerTime': data.get('lastTriggerTime'),
            'created': data.get('createdAt'),
            'updated': data.get('updatedAt')
        },
        'timestamp': timestamp
    }
    print(json.dumps(result, indent=2))
except Exception as e:
    print(json.dumps({'success': False, 'error': str(e), 'timestamp': timestamp}, indent=2))
    sys.exit(1)
"
