---
name: pyzotero-cli
version: 2.2.0
description: Python scripts for Zotero - supports search, browse, add items, and full collection management. Both local API and online Web API modes.
homepage: https://github.com/urschrei/pyzotero
metadata:
  {
    "openclaw":
      {
        "emoji": "📚",
        "requires": { "anyBins": ["pyzotero"], "bins": ["python3"] },
        "install":
          [
            {
              "id": "pipx_lib",
              "kind": "pipx",
              "package": "pyzotero",
              "label": "Install pyzotero library (pipx - recommended)",
              "platforms": ["linux-debian", "linux-ubuntu", "linux-arch", "linux-fedora", "linux-rhel"],
            },
            {
              "id": "pip_lib",
              "kind": "pip",
              "package": "pyzotero",
              "label": "Install pyzotero library (pip)",
            },
          ],
      },
  }
---

# Pyzotero CLI - Python Scripts

使用 Python 脚本调用 pyzotero 库，支持本地 Zotero API 和在线 Web API 两种模式。

## 快速开始

```bash
# 安装 pyzotero 库
pipx install pyzotero

# 设置环境变量 (可选)
export ZOTERO_LOCAL="true"  # 使用本地 API (默认)
# export ZOTERO_LOCAL="false"  # 使用在线 API

# 搜索库
python3 scripts/zotero_tool.py search -q "machine learning"

# 全文搜索 (包括 PDF)
python3 scripts/zotero_tool.py search -q "attention mechanisms" --fulltext

# 列出所有集合
python3 scripts/zotero_tool.py listcollections
```

📖 **详细指南:** [QUICKSTART.md](QUICKSTART.md)

## 环境变量配置

### ZOTERO_LOCAL (必需)
控制使用本地 API 还是在线 API:

| 值 | 模式 | 说明 |
|---|---|---|
| `"true"` (默认) | 本地模式 | 使用本地 Zotero 7+ 的本地 API |
| `"false"` | 在线模式 | 使用 Zotero Web API |

### ZOTERO_USER_ID (在线模式必需)
您的 Zotero 用户 ID，在在线模式下需要设置。

### ZOTERO_API_KEY (在线模式必需)
您的 Zotero API Key，在在线模式下需要设置。

### 配置示例

**本地模式 (推荐):**
```bash
export ZOTERO_LOCAL="true"
python3 scripts/zotero_tool.py search -q "python"
```

**在线模式:**
```bash
export ZOTERO_LOCAL="false"
export ZOTERO_USER_ID="your_user_id"
export ZOTERO_API_KEY="your_api_key"
python3 scripts/zotero_tool.py search -q "python"
```

## 安装

### pipx (推荐)
```bash
pipx install pyzotero
```

### pip (通用)
```bash
pip install --user pyzotero
export PATH="$HOME/.local/bin:$PATH"
```

📖 **完整安装指南:** [INSTALL.md](INSTALL.md)

## 前提条件

### 本地模式配置

**需要在 Zotero 7+ 中启用本地 API:**

1. 打开 Zotero 7 (或更新版本)
2. 进入 **编辑 > 首选项 > 高级** (macOS: **Zotero > 设置 > 高级**)
3. 勾选 **"允许此计算机上的其他应用程序与 Zotero 通信"**
4. 重启 Zotero

### 在线模式配置

**需要获取 API 密钥:**

1. 访问 https://www.zotero.org/settings/keys
2. 点击 "Create new private key"
3. 授予读取权限 (Read access to library and files)
4. 复制密钥并设置环境变量:
   ```bash
   export ZOTERO_USER_ID="your_user_id"
   export ZOTERO_API_KEY="your_key"
   ```

## 核心命令

| 命令 | 说明 |
|------|------|
| `python3 scripts/zotero_tool.py search -q "关键词"` | 搜索库 |
| `python3 scripts/zotero_tool.py search --fulltext` | 全文搜索 (包括 PDF) |
| `python3 scripts/zotero_tool.py search --collection ID` | 在特定集合中搜索 |
| `python3 scripts/zotero_tool.py listcollections` | 列出所有集合 |
| `python3 scripts/zotero_tool.py itemtypes` | 列出项目类型 |
| `python3 scripts/zotero_tool.py item KEY` | 获取单个项目详情 |
| `python3 scripts/zotero_tool.py add -t "标题"` | **添加单个项目** |
| `python3 scripts/zotero_tool.py add-from-json file.json` | **批量添加项目** |
| `python3 scripts/zotero_tool.py collection-create -n "名称"` | **创建新集合** |
| `python3 scripts/zotero_tool.py collection-rename KEY -n "新名"` | **重命名集合** |
| `python3 scripts/zotero_tool.py collection-delete KEY -y` | **删除集合** |
| `python3 scripts/zotero_tool.py collection-add-item ITEM -c COLL` | **添加项目到集合** |
| `python3 scripts/zotero_tool.py collection-remove-item ITEM -c COLL` | **从集合移除项目** |
| `python3 scripts/zotero_tool.py collection-list KEY` | **列出集合中的项目** |

## 使用示例

### 基本搜索
```bash
# 搜索标题和元数据
python3 scripts/zotero_tool.py search -q "machine learning"

# 短语搜索
python3 scripts/zotero_tool.py search -q "\"deep learning\""
```

### 全文搜索
```bash
# 在 PDF 和附件中搜索
python3 scripts/zotero_tool.py search -q "neural networks" --fulltext
```

### 高级过滤
```bash
# 按项目类型过滤
python3 scripts/zotero_tool.py search -q "methodology" --itemtype book

# 在特定集合中搜索
python3 scripts/zotero_tool.py search --collection ABC123 -q "test"

# 限制结果数量
python3 scripts/zotero_tool.py search -q "python" -l 10
```

### 获取项目详情
```bash
# 获取单个项目
python3 scripts/zotero_tool.py item ABC123XYZ
```

### 添加单个项目
```bash
# 添加期刊文章
python3 scripts/zotero_tool.py add -t "文章标题" -a "FirstName LastName" -p "期刊名" -d "2024"

# 添加带 DOI 和标签的文章
python3 scripts/zotero_tool.py add -t "AI 在医学中的应用" -a "张三" "李四" -p "中华医学杂志" -d "2024" --doi "10.xxxx/xxxxx" --tags AI 医学 machine-learning

# 添加带摘要和 URL 的文章
python3 scripts/zotero_tool.py add -t "Python 数据分析" -a "王五" -p "计算机学报" -d "2024" --url "https://example.com" --abstract "本文介绍了..." --tags python 数据分析
```

### 批量添加项目
```bash
# 从 JSON 文件批量添加
python3 scripts/zotero_tool.py add-from-json papers.json

# JSON 文件格式示例:
[
  {
    "itemType": "journalArticle",
    "title": "文章标题",
    "creators": [{"firstName": "First", "lastName": "Last", "creatorType": "author"}],
    "publicationTitle": "期刊名",
    "date": "2024",
    "DOI": "10.xxxx/xxxxx",
    "tags": [{"tag": "tag1", "type": 1}],
    "abstractNote": "摘要"
  }
]
```

### 集合管理
```bash
# 创建新集合
python3 scripts/zotero_tool.py collection-create -n "新集合名称"

# 创建子集合
python3 scripts/zotero_tool.py collection-create -n "子集合" -p PARENT_KEY

# 重命名集合
python3 scripts/zotero_tool.py collection-rename COLLECTION_KEY -n "新名称"

# 删除集合（需要确认）
python3 scripts/zotero_tool.py collection-delete COLLECTION_KEY
python3 scripts/zotero_tool.py collection-delete COLLECTION_KEY -y  # 跳过确认

# 添加项目到集合
python3 scripts/zotero_tool.py collection-add-item ITEM_KEY -c COLLECTION_KEY

# 从集合移除项目
python3 scripts/zotero_tool.py collection-remove-item ITEM_KEY -c COLLECTION_KEY

# 列出集合中的项目
python3 scripts/zotero_tool.py collection-list COLLECTION_KEY
python3 scripts/zotero_tool.py collection-list COLLECTION_KEY -l 50  # 显示 50 个
```

## 输出格式

### 人类可读格式 (默认)
```bash
python3 scripts/zotero_tool.py search -q "python"
```

### JSON 输出
```bash
# 输出 JSON 格式
python3 scripts/zotero_tool.py search -q "topic" --json

# 使用 jq 处理
python3 scripts/zotero_tool.py search -q "topic" --json | jq '.[].data.title'

# 保存到文件
python3 scripts/zotero_tool.py search -q "topic" --json > results.json
```

## 文档

| 文档 | 说明 |
|------|------|
| [QUICKSTART.md](QUICKSTART.md) | 5 分钟快速入门 |
| [INSTALL.md](INSTALL.md) | 详细安装指南 |
| [EXAMPLES.md](EXAMPLES.md) | 实用使用示例 |
| [README.md](README.md) | 项目概览 |

## 故障排除

**本地模式连接错误:**
```
确保 Zotero 正在运行
启用本地 API: 设置 > 高级 > "允许此计算机上的其他应用程序与 Zotero 通信"
重启 Zotero
```

**在线模式认证错误:**
```bash
# 检查环境变量是否正确设置
echo $ZOTERO_USER_ID
echo $ZOTERO_API_KEY

# 确认 ZOTERO_LOCAL 设置为 false
export ZOTERO_LOCAL="false"
```

**模块未找到错误:**
```bash
# 确保已安装 pyzotero
pipx install pyzotero
# 或
pip install --user pyzotero
```

**权限错误 (PEP 668 系统):**
```bash
pipx install pyzotero
```

📖 **详细故障排除:** [INSTALL.md](INSTALL.md)

## 快速参考

```bash
# 设置模式
export ZOTERO_LOCAL="true"   # 本地模式 (默认)
export ZOTERO_LOCAL="false"  # 在线模式

# 在线模式需要额外设置
export ZOTERO_USER_ID="your_id"
export ZOTERO_API_KEY="your_key"

# 搜索
python3 scripts/zotero_tool.py search -q "topic"
python3 scripts/zotero_tool.py search -q "topic" --fulltext
python3 scripts/zotero_tool.py search -q "topic" --json

# 列表
python3 scripts/zotero_tool.py listcollections
python3 scripts/zotero_tool.py itemtypes

# 获取项目
python3 scripts/zotero_tool.py item ITEM_KEY

# 过滤
python3 scripts/zotero_tool.py search -q "topic" --itemtype journalArticle
python3 scripts/zotero_tool.py search --collection ABC123 -q "topic"
```

---

**完整文档:**
- [QUICKSTART.md](QUICKSTART.md) - 快速入门
- [INSTALL.md](INSTALL.md) - 安装详情
- [EXAMPLES.md](EXAMPLES.md) - 使用示例
- [README.md](README.md) - 完整概览

## 脚本说明

本技能提供 Python 脚本 `scripts/zotero_tool.py`，通过 pyzotero 库与 Zotero 交互:

- **本地模式** (`ZOTERO_LOCAL="true"`): 直接连接到运行中的 Zotero 7+ 本地实例
- **在线模式** (`ZOTERO_LOCAL="false"`): 通过 Zotero Web API 访问您的在线库

所有功能与原版 pyzotero-cli 保持一致，但使用 Python 脚本方式调用，更灵活且易于集成到其他 Python 项目中。
