import { useState } from "react";

/**
 * Represents a single task item generated by the agent.
 * Each task item can be of different types such as tool calls, notifications, or errors.
 *
 * @property type - The type of the task item, defined by the AgentTaskItemType enum.
 * @property messageId - The unique identifier for the task item.
 * This is an incremental number representing the n-th task item.
 * @property data - The data associated with the task item, which varies based on the type.
 * @property isFinal - (Optional) Indicates if the task item is final and no further updates are expected.
 */
export type AgentTaskMessage = {
  type: AgentTaskMessageType;
  messageId: number; // The incremental ID of the task item
  data: AgentTaskMessageData;
  isFinal?: boolean;
};

export type AgentTaskMessageUpdate = {
  type: AgentTaskMessageType;
  messageId: number;
  // updateType: "append" | "replace";
  // "replace" type is not supported currently
  updateType: "append";
  delta: AgentTaskMessageData;
  isFinal?: boolean;
  timeUsedSec?: string;
};

export enum AgentTaskMessageType {
  Creation = "creation",
  Update = "update",
}

export enum AgentTaskMessageDataType {
  Notification = "notification",
  ToolCall = "toolCall",
  ArtifactOutput = "artifactOutput",
}

/**
 * Data associated with an AgentTaskItem.
 * The fields included depend on the type of the task item.
 *
 * @property type - The type of the task item, defined by the AgentTaskItemType enum.
 * @property title - (Optional) A brief title or summary of the task item.
 * @property description - (Optional) A detailed description of the task item.
 * @property toolName - (Optional) The name of the tool being called (if applicable).
 * @property parameters - (Optional) A record of parameters associated with the tool call (if applicable).
 * @property error - (Optional) An error message if the task item represents an error.
 * @property result - (Optional) The result or output of the task item (if applicable).
 */
export type AgentTaskMessageData = {
  type?: AgentTaskMessageDataType;
  title?: string;
  description?: string;
  toolName?: string;
  parameters?: Record<string, unknown>;
  error?: string;
  result?: string;
};

export default function useCodeGenV2() {
  const [messages, setMessages] = useState<AgentTaskMessage[]>([]);
  const [isGenerating, setIsGenerating] = useState(false);

  async function generateCode({
    prompt,
    appName,
    appId,
    appVersion,
  }: {
    prompt: string;
    appName?: string;
    appId?: string;
    appVersion?: string;
  }) {
    setIsGenerating(true);
    const response = await fetch("/server-function/generate-code/v2/generate", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        prompt,
        appName,
        appId,
        version: appVersion,
      }),
    });

    if (!response.ok) {
      setIsGenerating(false);
      throw new Error(`Server error: ${await response.text()}`);
    }

    const stream = await response.body;
    if (!stream) {
      setIsGenerating(false);
      throw new Error("No stream returned from server");
    }

    const reader = stream.getReader();
    const decoder = new TextDecoder();
    // SSE buffer
    let buffer = "";

    const collectedMessages: AgentTaskMessage[] = [];

    try {
      while (true) {
        const { value, done } = await reader.read();
        if (done) break;

        buffer += decoder.decode(value, { stream: true });

        // SSE messages end with \n\n
        const parts = buffer.split("\n\n");
        // Only keep the last part in the buffer (it may be incomplete)
        buffer = parts.pop()!;

        for (const part of parts) {
          if (!part.startsWith("data:")) continue;

          const json = part.replace(/^data:\s*/, "");

          // Json is either AgentTaskMessage or AgentTaskMessageUpdate
          const message = JSON.parse(json) as
            | AgentTaskMessage
            | AgentTaskMessageUpdate;

          if (message.type === AgentTaskMessageType.Creation) {
            // New message
            collectedMessages.push(message as AgentTaskMessage);
            setMessages([...collectedMessages]);
            continue;
          } else if (message.type === AgentTaskMessageType.Update) {
            // Update existing message
            const update = message as AgentTaskMessageUpdate;
            const msgIndex = collectedMessages.findIndex(
              (msg) => msg.messageId === update.messageId,
            );
            if (msgIndex !== -1) {
              // Append delta to existing message data
              collectedMessages[msgIndex].data.result =
                (collectedMessages[msgIndex].data.result ?? "") +
                (update.delta.result ?? "");

              collectedMessages[msgIndex].data.error =
                (collectedMessages[msgIndex].data.error ?? "") +
                (update.delta.error ?? "");

              collectedMessages[msgIndex].isFinal = update.isFinal;
              setMessages([...collectedMessages]);
            }
          }
        }
      }
    } catch (e) {
      console.error("Error reading stream:", e);
      throw e;
    } finally {
      setIsGenerating(false);
    }

    // Check if any message contains a artifact output with publish link
    const artifactMessage = collectedMessages.find((msg) => {
      return msg.data.type === AgentTaskMessageDataType.ArtifactOutput;
    });

    console.log("Artifact message:", artifactMessage);

    if (artifactMessage) {
      const json = artifactMessage.data.result!;
      const parsed = JSON.parse(json) as {
        publishedAppLink?: string;
        sourceCodeArchiveLink?: string;
        appId: string;
        version?: string;
      };
      return parsed;
    }
    return undefined;
  }

  return { messages, isGenerating, generateCode };
}
