#!/usr/bin/env python3
"""
Protoss Voice Effect Generator (Khala Engine)
---------------------------------------------
Applies a psionic audio effect to voice recordings using SoX and FFmpeg.
The effect mimics the StarCraft Protoss voice style: deep, resonant, and telepathic.

Architecture: "The Void" (V9)
- Removes the original human voice entirely.
- Processes the signal into a deep, reversed-reverb "shadow" entity.
- Results in a distant, massive, and telepathic sound signature.

Dependencies:
- ffmpeg
- sox
"""

import subprocess
import sys
import os
import shutil

def check_dependencies():
    """Verify that required tools are installed."""
    missing = []
    if not shutil.which("ffmpeg"):
        missing.append("ffmpeg")
    if not shutil.which("sox"):
        missing.append("sox")

    if missing:
        print(f"❌ Error: Missing required dependencies: {', '.join(missing)}")
        print("Please install them using brew:")
        print(f"  brew install {' '.join(missing)}")
        return False
    return True

def apply_protoss_effect(input_path, output_path):
    """
    Apply the Protoss psionic effect chain to an audio file.

    Args:
        input_path (str): Path to input audio file.
        output_path (str): Path where the processed file will be saved.
    """
    if not check_dependencies():
        return

    print(f"🔮 Channeling Protoss Psionic Energy to: {input_path}")

    # Define temporary file paths using absolute paths to avoid confusion
    work_dir = os.path.dirname(os.path.abspath(output_path))
    filename_base = os.path.splitext(os.path.basename(output_path))[0]

    temp_wav_in = os.path.join(work_dir, f".{filename_base}_in.wav")
    temp_wav_psi = os.path.join(work_dir, f".{filename_base}_psi.wav")
    temp_wav_mix = os.path.join(work_dir, f".{filename_base}_mix.wav")

    try:
        # Step 1: Convert input to WAV (canonical format for SoX)
        # Using ffmpeg to handle any weird input formats (mp3, ogg, etc.)
        subprocess.run(
            ["ffmpeg", "-y", "-i", input_path, "-vn", temp_wav_in],
            check=True,
            stdout=subprocess.DEVNULL,
            stderr=subprocess.DEVNULL
        )

        # Step 2: Generate the Psionic Layer (The "Shadow")
        # Transformation Chain:
        # 1. pitch -200: Drops pitch 2 semitones WITHOUT slowing down (replaces speed 0.9).
        # 2. reverse:    Flips audio to prepare for pre-verb.
        # 3. reverb:     Applies massive hall reverb (80% size, 100% scale).
        # 4. reverse:    Flips back. The reverb tails now precede the voice ("sucking" effect).
        # 5. vol 0.5:    Gain staging to prevent clipping in the next stage.
        cmd_psi = [
            "sox",
            temp_wav_in,
            temp_wav_psi,
            "pitch", "-200",
            "reverse",
            "reverb", "80", "50", "100", "100", "0", "0",
            "reverse",
            "vol", "0.5"
        ]
        subprocess.run(cmd_psi, check=True)

        # Step 3: Mastering ("The Void")
        # We discard the dry vocal completely. Only the processed shadow remains.
        # 1. bass +5:    Restores body lost during reverb diffusion.
        # 2. highpass:   Cleans up subsonic mud below 100Hz.
        # 3. norm -1:    Maximizes volume without clipping (-1dB headroom).
        cmd_mix = [
            "sox",
            temp_wav_psi,
            temp_wav_mix,
            "bass", "+5",
            "highpass", "100",
            "norm", "-1"
        ]
        subprocess.run(cmd_mix, check=True)

        # Step 4: Encode final output (MP3/Original format)
        subprocess.run(
            ["ffmpeg", "-y", "-i", temp_wav_mix, output_path],
            check=True,
            stdout=subprocess.DEVNULL,
            stderr=subprocess.DEVNULL
        )

        print(f"✨ Psionic Entity Stabilized: {output_path}")

    except subprocess.CalledProcessError as e:
        print(f"❌ Disruption in the Khala (Processing Error): {e}")
    except Exception as e:
        print(f"❌ Unknown Error: {e}")
    finally:
        # Cleanup temporary artifacts
        for temp_file in [temp_wav_in, temp_wav_psi, temp_wav_mix]:
            if os.path.exists(temp_file):
                try:
                    os.remove(temp_file)
                except OSError:
                    pass

if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: python3 protoss_fx.py <input_audio_file>")
        sys.exit(1)

    input_file = sys.argv[1]

    # Auto-generate output filename if not provided
    # input.mp3 -> input_psionic.mp3
    base_name, ext = os.path.splitext(input_file)
    output_file = f"{base_name}_psionic{ext}"

    apply_protoss_effect(input_file, output_file)
