#!/usr/bin/env python3
"""Send an email via Proton Mail Bridge (localhost SMTP), using age-encrypted config.

Encrypted config:
  ~/clawd/secrets/proton.env.age
Age identity:
  ~/.config/age/keys.txt

Config keys supported:
- PROTON_EMAIL
- PROTON_BRIDGE_USER
- PROTON_BRIDGE_PASS
- SMTP_HOST, SMTP_PORT
- SMTP_SECURITY (STARTTLS|SSL|NONE)

Usage:
  send_email.py --to you@example.com --subject "Hi" --body-file /path/body.txt
  send_email.py --to you@example.com --subject "Hi" --body "Hello"
"""

import argparse
import os
import ssl
import subprocess
import smtplib
from email.message import EmailMessage


def decrypt_age(path: str, key_path: str) -> str:
    return subprocess.check_output(["age", "-d", "-i", key_path, path], text=True)


def parse_env(text: str) -> dict:
    conf = {}
    for line in text.splitlines():
        line = line.strip()
        if not line or line.startswith("#") or "=" not in line:
            continue
        k, v = line.split("=", 1)
        conf[k.strip()] = v.strip()
    return conf


def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--to", required=True)
    ap.add_argument("--subject", required=True)
    body_group = ap.add_mutually_exclusive_group(required=True)
    body_group.add_argument("--body")
    body_group.add_argument("--body-file")
    ap.add_argument("--from")

    args = ap.parse_args()

    enc_path = os.path.expanduser("~/clawd/secrets/proton.env.age")
    key_path = os.path.expanduser("~/.config/age/keys.txt")

    plain = decrypt_age(enc_path, key_path)
    conf = parse_env(plain)

    smtp_host = conf.get("SMTP_HOST", "127.0.0.1")
    smtp_port = int(conf.get("SMTP_PORT", "1025"))
    smtp_user = conf.get("PROTON_BRIDGE_USER") or conf.get("PROTON_EMAIL")
    smtp_pass = conf.get("PROTON_BRIDGE_PASS")
    sec = (conf.get("SMTP_SECURITY") or "").upper()

    from_addr = getattr(args, "from") or conf.get("PROTON_EMAIL") or smtp_user

    if args.body_file:
        with open(args.body_file, "r", encoding="utf-8") as f:
            body = f.read()
    else:
        body = args.body

    msg = EmailMessage()
    msg["From"] = from_addr
    msg["To"] = args.to
    msg["Subject"] = args.subject
    msg.set_content(body)

    # Proton Bridge uses a local cert; for localhost automation we allow it.
    local_ctx = ssl._create_unverified_context()

    if "SSL" in sec and "START" not in sec:
        with smtplib.SMTP_SSL(smtp_host, smtp_port, context=local_ctx, timeout=30) as s:
            if smtp_user and smtp_pass:
                s.login(smtp_user, smtp_pass)
            s.send_message(msg)
    elif "START" in sec:
        with smtplib.SMTP(smtp_host, smtp_port, timeout=30) as s:
            s.ehlo()
            s.starttls(context=local_ctx)
            s.ehlo()
            if smtp_user and smtp_pass:
                s.login(smtp_user, smtp_pass)
            s.send_message(msg)
    else:
        with smtplib.SMTP(smtp_host, smtp_port, timeout=30) as s:
            s.ehlo()
            if smtp_user and smtp_pass:
                s.login(smtp_user, smtp_pass)
            s.send_message(msg)


if __name__ == "__main__":
    main()
