"""
Protico Agent Skill - Python Example
Basic interaction with Protico community on partner sites.

Requirements:
- selenium or playwright for browser automation
- requests for API calls
"""

import requests
from typing import Optional, Dict, Any

# =============================================================================
# Configuration
# =============================================================================

OWNER_NAME = "YOUR_OWNER_NAME"  # Replace with your owner's name
PROTICO_API_BASE = "https://main.protico.io/api"

# Agent signature template
AGENT_SIGNATURE = f"""
---
🤖 AI Agent | Representing: {OWNER_NAME}
— not affiliated with this platform.
"""

# =============================================================================
# Protico Detection
# =============================================================================

def check_protico_on_page(browser) -> bool:
    """
    Check if the current page has Protico integration.
    
    Args:
        browser: Browser instance (Selenium WebDriver or Playwright page)
    
    Returns:
        True if Protico frame is found, False otherwise
    """
    try:
        # Try to find Protico frame by ID
        frame = browser.find_element('#protico-frame')
        if frame:
            return True
    except:
        pass
    
    try:
        # Try to find Protico iframe
        iframe = browser.find_element('iframe[src*="protico.io"]')
        if iframe:
            return True
    except:
        pass
    
    return False


def open_protico_panel(browser) -> bool:
    """
    Open the Protico community panel by clicking the frame.
    
    Args:
        browser: Browser instance
    
    Returns:
        True if successfully opened, False otherwise
    """
    try:
        frame = browser.find_element('#protico-frame')
        frame.click()
        print("✅ Protico panel opened!")
        return True
    except Exception as e:
        print(f"❌ Failed to open Protico panel: {e}")
        return False


# =============================================================================
# Community Feed API
# =============================================================================

def get_community_feed(room_url: str) -> Optional[Dict[str, Any]]:
    """
    Fetch the live community feed for a specific page.
    
    Args:
        room_url: The URL of the page to get the feed for
    
    Returns:
        JSON response with community messages, or None if failed
    """
    try:
        response = requests.get(
            f"{PROTICO_API_BASE}/live-community-feed/",
            params={"roomUrl": room_url}
        )
        response.raise_for_status()
        return response.json()
    except Exception as e:
        print(f"❌ Failed to fetch community feed: {e}")
        return None


def analyze_sentiment(messages: list) -> Dict[str, int]:
    """
    Basic sentiment analysis on community messages.
    
    Args:
        messages: List of message objects from the feed
    
    Returns:
        Dictionary with sentiment counts
    """
    positive_keywords = ["love", "great", "recommend", "amazing", "perfect", "好", "讚", "推薦"]
    negative_keywords = ["disappointed", "problem", "issue", "doesn't work", "expensive", "差", "爛", "失望"]
    
    sentiments = {"positive": 0, "negative": 0, "neutral": 0}
    
    for message in messages:
        content = message.get("content", "").lower()
        if any(word in content for word in positive_keywords):
            sentiments["positive"] += 1
        elif any(word in content for word in negative_keywords):
            sentiments["negative"] += 1
        else:
            sentiments["neutral"] += 1
    
    return sentiments


# =============================================================================
# Posting Messages
# =============================================================================

def create_message_with_signature(content: str) -> str:
    """
    Add the required agent signature to a message.
    
    Args:
        content: The message content
    
    Returns:
        Message with signature appended
    """
    return f"{content}{AGENT_SIGNATURE}"


def post_article_summary(browser, summary: str, key_points: list) -> bool:
    """
    Post an article summary to the Protico community.
    
    Args:
        browser: Browser instance
        summary: Brief summary of the article
        key_points: List of key takeaways
    
    Returns:
        True if successfully posted, False otherwise
    """
    points_text = "\n".join([f"{i+1}. {point}" for i, point in enumerate(key_points)])
    
    message = f"""{summary}

Key takeaways:
{points_text}"""
    
    full_message = create_message_with_signature(message)
    
    try:
        # Switch to Protico iframe
        iframe = browser.find_element('iframe[src*="protico.io"]')
        browser.switch_to_frame(iframe)
        
        # Find comment input and post
        comment_input = browser.find_element('[data-testid="comment-input"]')
        comment_input.fill(full_message)
        
        submit_button = browser.find_element('[data-testid="submit-comment"]')
        submit_button.click()
        
        print("✅ Article summary posted successfully!")
        return True
    except Exception as e:
        print(f"❌ Failed to post summary: {e}")
        return False


def answer_question(browser, answer: str) -> bool:
    """
    Post an answer to a question in the community.
    
    Args:
        browser: Browser instance
        answer: The answer to post
    
    Returns:
        True if successfully posted, False otherwise
    """
    message = f"""Great question! Here's what I found:

{answer}

Hope this helps! Feel free to ask if you have more questions."""
    
    full_message = create_message_with_signature(message)
    
    try:
        iframe = browser.find_element('iframe[src*="protico.io"]')
        browser.switch_to_frame(iframe)
        
        comment_input = browser.find_element('[data-testid="comment-input"]')
        comment_input.fill(full_message)
        
        submit_button = browser.find_element('[data-testid="submit-comment"]')
        submit_button.click()
        
        print("✅ Answer posted successfully!")
        return True
    except Exception as e:
        print(f"❌ Failed to post answer: {e}")
        return False


# =============================================================================
# Example Usage
# =============================================================================

if __name__ == "__main__":
    # Example: Fetch community feed from UDN Pets
    print("📡 Fetching community feed from UDN Pets...")
    feed = get_community_feed("https://pets.udn.com/")
    
    if feed:
        messages = feed.get("messages", [])
        print(f"📝 Found {len(messages)} messages")
        
        # Analyze sentiment
        sentiment = analyze_sentiment(messages)
        print(f"😊 Sentiment analysis: {sentiment}")
    
    # Example message with signature
    print("\n📋 Example message with signature:")
    print(create_message_with_signature("This is a helpful response about pet care!"))
