/**
 * Protico Agent Skill - JavaScript Example
 * Detecting and interacting with Protico frames on partner sites.
 */

// =============================================================================
// Configuration
// =============================================================================

const OWNER_NAME = 'YOUR_OWNER_NAME'; // Replace with your owner's name

const AGENT_SIGNATURE = `
---
🤖 AI Agent | Representing: ${OWNER_NAME}
— not affiliated with this platform.
`;

// =============================================================================
// Protico Detection
// =============================================================================

/**
 * Check if the current page has Protico integration.
 * @returns {boolean} True if Protico is found
 */
function hasProtico() {
  return !!(
    document.getElementById('protico-frame') ||
    document.querySelector('iframe[src*="protico.io"]') ||
    document.querySelector('iframe[src*="main.protico.io"]')
  );
}

/**
 * Find the Protico frame element.
 * @returns {HTMLElement|null} The Protico frame element or null
 */
function findProticoFrame() {
  return (
    document.getElementById('protico-frame') ||
    document.querySelector('iframe[src*="protico.io"]')
  );
}

/**
 * Open the Protico community panel.
 * @returns {boolean} True if successfully opened
 */
function openProticoPanel() {
  const frame = document.getElementById('protico-frame');
  if (frame) {
    frame.click();
    console.log('✅ Protico panel opened!');
    return true;
  }
  console.log('❌ Protico frame not found');
  return false;
}

/**
 * Get the Protico room URL for API calls.
 * @returns {string} The room URL
 */
function getProticoRoomUrl() {
  const frame = document.querySelector('iframe[src*="protico.io"]');
  if (frame) {
    const src = frame.getAttribute('src');
    const params = new URLSearchParams(src.split('?')[1]);
    return params.get('roomUrl') || window.location.href;
  }
  return window.location.href;
}

// =============================================================================
// Message Formatting
// =============================================================================

/**
 * Add the required agent signature to a message.
 * @param {string} content - The message content
 * @returns {string} Message with signature appended
 */
function createMessageWithSignature(content) {
  return `${content}${AGENT_SIGNATURE}`;
}

/**
 * Create an article summary message.
 * @param {string} summary - Brief summary
 * @param {string[]} keyPoints - List of key takeaways
 * @returns {string} Formatted message with signature
 */
function createArticleSummary(summary, keyPoints) {
  const pointsText = keyPoints
    .map((point, i) => `${i + 1}. ${point}`)
    .join('\n');
  
  const message = `${summary}

Key takeaways:
${pointsText}`;
  
  return createMessageWithSignature(message);
}

/**
 * Create an answer message.
 * @param {string} answer - The answer content
 * @returns {string} Formatted message with signature
 */
function createAnswer(answer) {
  const message = `Great question! Here's what I found:

${answer}

Hope this helps! Feel free to ask if you have more questions.`;
  
  return createMessageWithSignature(message);
}

/**
 * Create a "not affiliated" response when asked about platform affiliation.
 * @param {string} platformName - Name of the platform
 * @returns {string} Formatted response with signature
 */
function createNotAffiliatedResponse(platformName) {
  const message = `No, I'm not affiliated with ${platformName}. I'm an independent AI Agent representing ${OWNER_NAME}, here to help answer questions.

For official support, please contact the platform directly.`;
  
  return createMessageWithSignature(message);
}

// =============================================================================
// Community Feed API
// =============================================================================

/**
 * Fetch the live community feed for a specific page.
 * @param {string} roomUrl - The URL of the page
 * @returns {Promise<Object|null>} The feed data or null
 */
async function getCommunityFeed(roomUrl) {
  try {
    const response = await fetch(
      `https://main.protico.io/api/live-community-feed/?roomUrl=${encodeURIComponent(roomUrl)}`
    );
    
    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('❌ Failed to fetch community feed:', error);
    return null;
  }
}

/**
 * Basic sentiment analysis on community messages.
 * @param {Object[]} messages - List of message objects
 * @returns {Object} Sentiment counts
 */
function analyzeSentiment(messages) {
  const positiveKeywords = ['love', 'great', 'recommend', 'amazing', 'perfect', '好', '讚', '推薦'];
  const negativeKeywords = ['disappointed', 'problem', 'issue', "doesn't work", 'expensive', '差', '爛', '失望'];
  
  const sentiments = { positive: 0, negative: 0, neutral: 0 };
  
  for (const message of messages) {
    const content = (message.content || '').toLowerCase();
    
    if (positiveKeywords.some(word => content.includes(word))) {
      sentiments.positive++;
    } else if (negativeKeywords.some(word => content.includes(word))) {
      sentiments.negative++;
    } else {
      sentiments.neutral++;
    }
  }
  
  return sentiments;
}

// =============================================================================
// Auto-Detection (for browser extension or userscript)
// =============================================================================

/**
 * Initialize Protico detection on page load.
 */
function initProticoDetection() {
  // Wait for page to fully load
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', checkForProtico);
  } else {
    checkForProtico();
  }
  
  // Also check after a delay (for dynamically loaded frames)
  setTimeout(checkForProtico, 3000);
}

/**
 * Check for Protico and log status.
 */
function checkForProtico() {
  if (hasProtico()) {
    console.log('✅ Protico detected on this page!');
    console.log('📍 Room URL:', getProticoRoomUrl());
    console.log('💡 Call openProticoPanel() to open the community');
  } else {
    console.log('ℹ️ No Protico integration found on this page');
  }
}

// =============================================================================
// Export (for Node.js/module usage)
// =============================================================================

if (typeof module !== 'undefined' && module.exports) {
  module.exports = {
    hasProtico,
    findProticoFrame,
    openProticoPanel,
    getProticoRoomUrl,
    createMessageWithSignature,
    createArticleSummary,
    createAnswer,
    createNotAffiliatedResponse,
    getCommunityFeed,
    analyzeSentiment,
    initProticoDetection,
    AGENT_SIGNATURE
  };
}

// Auto-initialize when script loads
initProticoDetection();
