#!/usr/bin/env python3
"""
Prompt Assemble - Token-Safe Memory Orchestration Framework

Engineering-grade implementation for production use.
Handles all edge cases, type safety, and OpenClaw integration.

Usage:
    from prompt_assemble import PromptAssembler
    
    assembler = PromptAssembler(
        max_tokens=204000,  # MiniMax-M2.1 context window
        safety_margin=0.75,  # Conservative: 75%
        memory_top_k=3,
        memory_summary_max_lines=3
    )
    
    result = assembler.build(
        user_input="...",
        system_prompt="...",
        get_recent_dialog_fn=lambda n: [...],
        memory_search_fn=lambda query, top_k: [...]
    )

Context Window Reference (as of 2026-02-04):
- MiniMax-M2.1: 204,000 tokens (default)
- Claude 3.5 Sonnet: 200,000 tokens
- GPT-4o: 128,000 tokens

Safety Design:
- Safety margin: 75% (conservative, leaves 25% buffer)
- Better to underutilize capacity than to overflow
"""

from typing import Any, Callable, List, Optional, Union


# Type aliases
PromptPart = str
ContextParts = List[PromptPart]
MemoryResult = List[str]
DialogFn = Callable[[int], ContextParts]
MemorySearchFn = Callable[[str, int], MemoryResult]


class PromptAssembler:
    """
    Token-safe prompt assembly with memory orchestration.
    
    Guarantees no API failure due to token overflow by implementing:
    - Two-phase context construction
    - Memory safety valve
    - Hard limits on memory injection
    """
    
    # Memory trigger keywords (hard-coded for reliability)
    MEMORY_TRIGGERS = [
        "previously",
        "earlier we discussed",
        "do you remember",
        "as I mentioned before",
        "continuing from",
        "before we",
        "last time",
        "previously mentioned"
    ]
    
    def __init__(
        self,
        max_tokens: Optional[int] = 204000,  # MiniMax-M2.1 default
        safety_margin: float = 0.75,  # Conservative: 75%
        memory_top_k: int = 3,
        memory_summary_max_lines: int = 3
    ):
        """
        Initialize PromptAssembler.
        
        Args:
            max_tokens: Model context limit (default: 204000 for MiniMax-M2.1)
            safety_margin: Threshold for safety valve (0.75 = 75% of max_tokens)
            memory_top_k: Maximum memories to retrieve
            memory_summary_max_lines: Maximum lines per memory summary
        
        Safety Margin (Conservative Design):
            - At 75%: 204,000 * 0.75 = 153,000 tokens available
            - Leaves ~51,000 tokens buffer for:
                - Model overhead and system prompts
                - Token estimation error margin
                - Request spikes and edge cases
            - Better to waste some capacity than to overflow
        """
        self.max_tokens = max_tokens
        self.safety_margin = safety_margin
        self.memory_top_k = memory_top_k
        self.memory_summary_max_lines = memory_summary_max_lines
        self.safety_threshold = (
            max_tokens * safety_margin if max_tokens else None
        )
    
    def need_memory(self, user_input: str) -> bool:
        """
        Determine if current input requires memory retrieval.
        Triggers are hard-coded to prevent false positives/negatives.
        
        Args:
            user_input: Raw user input text
            
        Returns:
            True if memory retrieval is needed, False otherwise
        """
        input_lower = user_input.lower()
        for trigger in self.MEMORY_TRIGGERS:
            if trigger.lower() in input_lower:
                return True
        return False
    
    def _ensure_string(self, part: Any) -> str:
        """
        Convert any type to string safely.
        
        Args:
            part: Any input that should be a string
            
        Returns:
            String representation
        """
        if isinstance(part, str):
            return part
        elif isinstance(part, list):
            return "\n".join(self._ensure_string(item) for item in part)
        elif isinstance(part, dict):
            return str(part)
        else:
            return str(part)
    
    def _assemble_parts(self, parts: List[Any]) -> str:
        """
        Assemble prompt parts into a single string.
        
        Args:
            parts: List of prompt parts (strings, lists, or other)
            
        Returns:
            Assembled prompt string
        """
        # Convert all parts to strings first
        string_parts = [self._ensure_string(part) for part in parts if part]
        # Join with separators
        return "\n\n---\n\n".join(string_parts)
    
    def _summarize_memory(
        self,
        memory: str,
        max_lines: Optional[int] = None
    ) -> str:
        """
        Compress memory to specified line limit.
        Preserves core information, removes redundancy.
        
        Args:
            memory: Raw memory content
            max_lines: Maximum lines in summary (default: instance setting)
            
        Returns:
            Summarized memory string
        """
        max_lines = max_lines or self.memory_summary_max_lines
        
        # Split into lines and filter empty
        lines = [
            line.strip() 
            for line in memory.strip().split("\n") 
            if line.strip()
        ]
        
        # Return original if already within limit
        if len(lines) <= max_lines:
            return memory
        
        # Summary strategy: keep first line (title/key) + last lines
        summary_lines = [lines[0]] + lines[-(max_lines - 1):]
        return "\n".join(summary_lines)
    
    def _estimate_tokens(self, text: str) -> int:
        """
        Estimate token count for given text.
        Uses language-aware estimation with conservative multiplier.
        
        Args:
            text: Input text
            
        Returns:
            Estimated token count
        """
        if not text:
            return 0
        
        # Detect if contains non-ASCII (likely Chinese/multilingual)
        has_multilingual = any(ord(c) > 127 for c in text)
        
        if has_multilingual:
            # Chinese: conservative 1.5 chars per token
            return int(len(text) / 1.5)
        else:
            # English: conservative 3.5 chars per token
            return int(len(text) / 3.5)
    
    def _apply_safety_valve(
        self,
        base_parts: List[Any],
        estimated_tokens: int
    ) -> str:
        """
        Apply safety valve when token budget exceeded.
        
        Args:
            base_parts: Base context parts
            estimated_tokens: Estimated token count
            
        Returns:
            Assembled prompt with safety notice
        """
        notice = (
            "\n[System Notice] "
            "Relevant memory skipped due to token budget "
            f"(estimated: {estimated_tokens} tokens)."
        )
        safe_parts = list(base_parts) + [notice]
        return self._assemble_parts(safe_parts)
    
    def build(
        self,
        user_input: str,
        system_prompt: str,
        get_recent_dialog_fn: Optional[DialogFn] = None,
        memory_search_fn: Optional[MemorySearchFn] = None
    ) -> str:
        """
        Build token-safe prompt with optional memory retrieval.
        
        Phase 1: Minimal Context (always included)
        Phase 2: Memory Decision (if needed)
        Phase 3: Memory Retrieval (if triggered)
        Phase 4: Token Estimation
        Phase 5: Safety Valve (if exceeded)
        Phase 6: Final Assembly
        
        Args:
            user_input: Current user input
            system_prompt: System prompt string
            get_recent_dialog_fn: Function to get recent dialog (n) -> list
            memory_search_fn: Function to search memories (query, top_k) -> list
            
        Returns:
            Assembled prompt string ready for LLM
        """
        # Validate required inputs (allow empty string, not None)
        if user_input is None:
            raise ValueError("user_input cannot be None")
        if system_prompt is None:
            raise ValueError("system_prompt cannot be None")
        
        # Phase 1: Build minimal context
        recent_dialog = ""
        if get_recent_dialog_fn:
            try:
                dialog_parts = get_recent_dialog_fn(3)
                recent_dialog = self._assemble_parts(dialog_parts) if dialog_parts else ""
            except Exception:
                recent_dialog = ""
        
        base_parts = [
            self._ensure_string(system_prompt),
            recent_dialog,
            self._ensure_string(user_input)
        ]
        
        # Phase 2: Memory decision
        use_memory = (
            self.need_memory(user_input) and 
            memory_search_fn is not None
        )
        
        if not use_memory:
            return self._assemble_parts(base_parts)
        
        # Phase 3: Memory retrieval
        try:
            memories = memory_search_fn(
                query=user_input,
                top_k=self.memory_top_k
            )
        except Exception:
            memories = []
        
        if not memories:
            return self._assemble_parts(base_parts)
        
        # Summarize each memory
        summarized = [
            self._summarize_memory(mem) 
            for mem in memories
        ]
        
        # Phase 4: Token estimation
        full_parts = base_parts + summarized
        full_text = self._assemble_parts(full_parts)
        estimated = self._estimate_tokens(full_text)
        
        # Phase 5: Safety valve check
        if (
            self.safety_threshold is not None and
            estimated > self.safety_threshold
        ):
            warning = self._apply_safety_valve(base_parts, estimated)
            # Optionally raise warning for logging
            return warning
        
        # Phase 6: Final assembly
        return full_text


# Convenience function for simple usage
def build_prompt(
    user_input: str,
    system_prompt: str,
    get_recent_dialog_fn: Optional[DialogFn] = None,
    memory_search_fn: Optional[MemorySearchFn] = None,
    max_tokens: Optional[int] = 204000,  # MiniMax-M2.1 default
    safety_margin: float = 0.75,  # Conservative: 75%
    memory_top_k: int = 3
) -> str:
    """
    Convenience wrapper for PromptAssembler.build()
    
    Args:
        user_input: Current user input
        system_prompt: System prompt string
        get_recent_dialog_fn: Function to get recent dialog
        memory_search_fn: Function to search memories
        max_tokens: Model context limit
        safety_margin: Safety threshold
        memory_top_k: Max memories to retrieve
        
    Returns:
        Assembled prompt string
    """
    assembler = PromptAssembler(
        max_tokens=max_tokens,
        safety_margin=safety_margin,
        memory_top_k=memory_top_k
    )
    
    return assembler.build(
        user_input=user_input,
        system_prompt=system_prompt,
        get_recent_dialog_fn=get_recent_dialog_fn,
        memory_search_fn=memory_search_fn
    )


# ============ CLI Tests ============
if __name__ == "__main__":
    import sys
    
    def test_basic_types():
        """Test type safety with various input types."""
        print("Test 1: Basic type handling")
        
        assembler = PromptAssembler(max_tokens=200000)
        
        # Mixed types should work
        result = assembler.build(
            user_input="Test",
            system_prompt="You are a helpful assistant.",
            get_recent_dialog_fn=lambda n: ["User: hello", "Bot: hi"],
            memory_search_fn=lambda query, top_k: ["Memory: test info"]
        )
        print(f"✓ Mixed types: {len(result)} chars")
        
    def test_safety_valve():
        """Test safety valve with limited tokens."""
        print("\nTest 2: Safety valve")
        
        assembler = PromptAssembler(
            max_tokens=500,  # Low limit for testing
            safety_margin=0.5  # 50% threshold = 250 tokens
        )
        
        # Create content that exceeds threshold
        result = assembler.build(
            user_input="do you remember our previous discussion?",
            system_prompt="System prompt: " + "x" * 400,  # ~400 chars
            get_recent_dialog_fn=lambda n: ["Dialog: " + "y" * 400],
            memory_search_fn=lambda query, top_k: ["Memory: " + "z" * 400]
        )
        
        if "skipped due to token budget" in result:
            print("✓ Safety valve triggered correctly")
        else:
            print(f"✗ Safety valve not triggered, got {len(result)} chars")
    
    def test_memory_triggers():
        """Test memory trigger detection."""
        print("\nTest 3: Memory triggers")
        
        assembler = PromptAssembler()
        
        tests = [
            ("as I mentioned before", True),
            ("continuing from our last discussion", True),
            ("do you remember what we decided", True),
            ("previously we agreed", True),
            ("today's weather is nice", False),
            ("Build a website", False),
            ("last time we met", True),  # "last time" triggers
        ]
        
        for text, expected in tests:
            result = assembler.need_memory(text)
            status = "✓" if result == expected else "✗"
            print(f"  {status} '{text}': {result} (expected: {expected})")
    
    def test_empty_handling():
        """Test handling of empty and None inputs."""
        print("\nTest 4: Empty/None input handling")
        
        assembler = PromptAssembler()
        
        # Empty string should work (valid input)
        try:
            result = assembler.build(
                user_input="",
                system_prompt="You are helpful."
            )
            print(f"✓ Empty string handled: {len(result)} chars")
        except Exception as e:
            print(f"✗ Empty string failed: {e}")
        
        # None should fail
        try:
            result = assembler.build(
                user_input=None,
                system_prompt="You are helpful."
            )
            print(f"✗ None user_input should fail")
        except ValueError as e:
            print(f"✓ None user_input correctly rejected: {e}")
        
        try:
            result = assembler.build(
                user_input="Test",
                system_prompt=None
            )
            print(f"✗ None system_prompt should fail")
        except ValueError as e:
            print(f"✓ None system_prompt correctly rejected: {e}")
    
    def test_memory_summary():
        """Test memory summarization."""
        print("\nTest 5: Memory summarization")
        
        assembler = PromptAssembler(memory_summary_max_lines=3)
        
        long_memory = "\n".join([f"Line {i}" for i in range(1, 11)])
        summary = assembler._summarize_memory(long_memory)
        line_count = len([l for l in summary.split("\n") if l])
        
        print(f"✓ Summary reduced from 10 to {line_count} lines")
    
    # Run all tests
    print("=" * 50)
    print("PromptAssembler Engineering Tests")
    print("=" * 50)
    
    test_basic_types()
    test_safety_valve()
    test_memory_triggers()
    test_empty_handling()
    test_memory_summary()
    
    print("\n" + "=" * 50)
    print("All tests completed!")
    print("=" * 50)
