# Token Estimation Strategies

## Overview

Different approaches to estimating token count for prompt assembly.

---

## Simple Estimation (Recommended)

**Formula:** `tokens ≈ characters / 3.5`

```python
def estimate_tokens(text: str) -> int:
    return len(text) // 3.5
```

**Accuracy:** ±20% (sufficient for safety checks)

**Pros:**
- Simple, fast
- No external dependencies
- Works for mixed languages

**Cons:**
- Not precise
- Overestimates for English
- Underestimates for code

---

## BPE-Based Estimation

More accurate but requires library.

```python
import tiktoken

def estimate_tokens(text: str, model: str = "claude-3-5-sonnet-20241022") -> int:
    encoding = tiktoken.encoding_for_model(model)
    return len(encoding.encode(text))
```

**Accuracy:** ±5%

**Pros:**
- Accurate for specific model
- Handles special tokens

**Cons:**
- Requires `tiktoken` library
- Model-specific
- Slower

---

## Language-Aware Estimation

Adjust ratio based on detected language.

```python
def estimate_tokens(text: str) -> int:
    # Rough heuristics
    if any(ord(c) > 127 for c in text):  # Contains non-ASCII (likely Chinese)
        return len(text) // 2
    else:
        return len(text) // 4
```

**Accuracy:** ±15%

---

## Practical Recommendations

### For Safety Valve
Use **simple estimation** - precision not critical, just need a safety buffer.

### For Exact Limits
Use **tiktoken** - necessary when near model limits.

### For Mixed Content
Use **language-aware** - better accuracy for multilingual prompts.

---

## Model Limits Reference

| Model | Context Limit |
|-------|---------------|
| Claude 3.5 Sonnet | 200K |
| Claude 3 Opus | 200K |
| GPT-4o | 128K |
| MiniMax-M2.1 | 256K |

**Always set SAFETY_MARGIN = 0.85** to leave buffer for model variations.
