#!/usr/bin/env python3
"""
Health AI Data Importer (v3.0 - AutoSync + ZIP Support)
Parses Health Auto Export files (AutoSync .hae, ZIP, or JSON) to skill data format.
Optimized for Apple Watch Ultra 2 exports.
"""

import json
import os
import sys
import zipfile
import csv
import io
import shutil
import time
import tempfile
from datetime import datetime, timedelta
from pathlib import Path

# Paths
HEALTH_EXPORT_PATH = os.path.expanduser("~/Library/Mobile Documents/iCloud~com~ifunography~HealthExport/Documents")
AUTOSYNC_PATH = os.path.join(HEALTH_EXPORT_PATH, "AutoSync/HealthMetrics")
SKILL_DATA_PATH = Path(__file__).parent.parent / "data"
TEMP_DIR = Path("/tmp/health_ai_import")

# Metrics we care about (folder name -> our slug)
METRIC_FOLDERS = {
    "heart_rate": "heart_rate",
    "resting_heart_rate": "resting_heart_rate",
    "heart_rate_variability": "heart_rate_variability",
    "blood_oxygen": "blood_oxygen_saturation",
    "respiratory_rate": "respiratory_rate",
    "body_temperature": "body_temperature",
    "sleep_analysis": "sleep_analysis",
    "wheelchair_distance": "wheelchair_distance",
    "push_count": "push_count",
}

# Legacy mapping for ZIP/JSON files
METRIC_MAP = {
    "Heart Rate": "heart_rate",
    "Resting Heart Rate": "resting_heart_rate",
    "Heart Rate Variability SDNN": "heart_rate_variability",
    "Blood Oxygen Saturation": "blood_oxygen_saturation",
    "Respiratory Rate": "respiratory_rate",
    "Body Temperature": "body_temperature",
    "Sleep Analysis": "sleep_analysis",
    "Step Count": "steps",
    "Distance Wheelchair": "wheelchair_distance",
    "Push Count": "push_count"
}

def safe_read_file(path, timeout_sec=2):
    """Read file with timeout for iCloud sync locks."""
    import signal
    
    def timeout_handler(signum, frame):
        raise TimeoutError("File read timed out")
    
    old_handler = signal.signal(signal.SIGALRM, timeout_handler)
    signal.alarm(timeout_sec)
    
    try:
        with open(path, 'r', encoding='utf-8', errors='replace') as f:
            content = f.read()
        signal.alarm(0)
        return content
    except (OSError, TimeoutError):
        signal.alarm(0)
        return None
    finally:
        signal.signal(signal.SIGALRM, old_handler)

def clean_temp():
    if TEMP_DIR.exists():
        shutil.rmtree(TEMP_DIR)
    TEMP_DIR.mkdir(parents=True, exist_ok=True)

def parse_autosync(days_back=3):
    """Parse recent .hae files from AutoSync folder."""
    data = {"vitals": []}
    autosync_dir = Path(AUTOSYNC_PATH)
    
    if not autosync_dir.exists():
        return data
    
    # Get date range to look for
    today = datetime.now()
    date_strs = [(today - timedelta(days=i)).strftime("%Y%m%d") for i in range(days_back)]
    
    files_processed = 0
    files_locked = 0
    
    for folder_name, metric_slug in METRIC_FOLDERS.items():
        folder_path = autosync_dir / folder_name
        if not folder_path.exists():
            continue
            
        for date_str in date_strs:
            hae_file = folder_path / f"{date_str}.hae"
            if not hae_file.exists():
                continue
                
            content = safe_read_file(hae_file)
            if content is None:
                files_locked += 1
                continue
                
            files_processed += 1
            try:
                hae_data = json.loads(content)
                # HAE format: {"data": [{"date": "...", "qty": ..., "source": "..."}, ...]}
                for reading in hae_data.get("data", []):
                    data["vitals"].append({
                        "metric": metric_slug,
                        "value": reading.get("qty"),
                        "date": reading.get("date"),
                        "unit": hae_data.get("units", ""),
                        "source": reading.get("source", "Apple Watch Ultra 2")
                    })
            except json.JSONDecodeError:
                continue
    
    if files_locked > 0:
        print(f"⚠️  {files_locked} files locked by iCloud sync, will retry next run")
    
    return data, files_processed

def find_latest_zip():
    """Find the most recent ZIP export."""
    export_dir = Path(HEALTH_EXPORT_PATH)
    files = list(export_dir.glob("*.zip"))
    
    if not files:
        return None
    
    files.sort(key=lambda f: f.stat().st_mtime, reverse=True)
    return files[0]

def parse_zip(zip_path):
    """Extract and parse CSVs from a ZIP export."""
    data = {"vitals": []}
    
    try:
        with zipfile.ZipFile(zip_path, 'r') as z:
            for filename in z.namelist():
                if not filename.endswith('.csv'):
                    continue
                    
                metric_name = None
                for key, val in METRIC_MAP.items():
                    if key in filename:
                        metric_name = val
                        break
                
                if not metric_name:
                    continue

                with z.open(filename) as f:
                    content = io.TextIOWrapper(f)
                    reader = csv.DictReader(content)
                    for row in reader:
                        value = row.get("value") or row.get("qty")
                        date = row.get("startDate") or row.get("date")
                        unit = row.get("unit") or ""
                        
                        if value and date:
                            try:
                                val_float = float(value) if value.replace('.', '', 1).replace('-', '', 1).isdigit() else value
                            except:
                                val_float = value
                            data["vitals"].append({
                                "metric": metric_name,
                                "value": val_float,
                                "date": date,
                                "unit": unit,
                                "source": "Apple Watch Ultra 2"
                            })
    except OSError as e:
        print(f"⚠️  ZIP file locked: {e}")
        return {"vitals": []}
        
    return data

def merge_data(new_data):
    """Merge into vitals.json with dedup (atomic write)."""
    vitals_path = SKILL_DATA_PATH / "vitals.json"
    
    existing = {"readings": []}
    if vitals_path.exists():
        for attempt in range(5):
            try:
                with open(vitals_path) as f:
                    existing = json.load(f)
                break
            except OSError as e:
                if e.errno == 11 and attempt < 4:
                    time.sleep(0.5 * (attempt + 1))
                    continue
                raise
        
    existing_keys = {f"{r['metric']}_{r['date']}" for r in existing.get("readings", [])}
    added = 0
    
    for r in new_data["vitals"]:
        key = f"{r['metric']}_{r['date']}"
        if key not in existing_keys:
            existing["readings"].append(r)
            existing_keys.add(key)
            added += 1
            
    existing["readings"].sort(key=lambda x: x["date"], reverse=True)
    existing["lastUpdated"] = datetime.now().isoformat()
    
    # Atomic write
    temp_fd, temp_path = tempfile.mkstemp(dir=SKILL_DATA_PATH, suffix='.json')
    try:
        with os.fdopen(temp_fd, 'w') as f:
            json.dump(existing, f, indent=2)
        os.replace(temp_path, vitals_path)
    except Exception:
        if os.path.exists(temp_path):
            os.unlink(temp_path)
        raise
        
    return added

def main():
    print("🏥 Health AI Import (v3)")
    clean_temp()
    
    total_added = 0
    
    # Try AutoSync first (preferred, more recent data)
    result = parse_autosync(days_back=3)
    if isinstance(result, tuple):
        autosync_data, files_processed = result
    else:
        autosync_data, files_processed = result, 0
    
    if autosync_data["vitals"]:
        print(f"📊 AutoSync: {len(autosync_data['vitals'])} readings from {files_processed} files")
        added = merge_data(autosync_data)
        total_added += added
    
    # Also check ZIP for any older data we might have missed
    zip_file = find_latest_zip()
    if zip_file:
        zip_data = parse_zip(zip_file)
        if zip_data["vitals"]:
            added = merge_data(zip_data)
            if added > 0:
                print(f"📦 ZIP: {added} additional readings from {zip_file.name}")
            total_added += added
    
    if total_added > 0:
        print(f"✅ Imported {total_added} new readings.")
    else:
        print("✓ No new readings.")

if __name__ == "__main__":
    main()
