#!/usr/bin/env python3
"""
Health Guardian - Analyze health data for patterns and anomalies.
"""

import json
import os
from datetime import datetime, timedelta
from pathlib import Path
from statistics import mean, stdev

def load_config():
    """Load configuration from config.json"""
    config_path = Path(__file__).parent.parent / "config.json"
    if not config_path.exists():
        config_path = Path(__file__).parent.parent / "config.example.json"
    with open(config_path) as f:
        return json.load(f)

def load_readings(data_dir: Path, days: int = 7):
    """Load health readings from the past N days"""
    readings_file = data_dir / "readings.json"
    if not readings_file.exists():
        return {}
    
    with open(readings_file) as f:
        all_readings = json.load(f)
    
    cutoff = datetime.now() - timedelta(days=days)
    filtered = {}
    
    for metric, values in all_readings.items():
        filtered[metric] = [
            v for v in values 
            if datetime.fromisoformat(v["timestamp"]) > cutoff
        ]
    
    return filtered

def calculate_baseline(values: list, baseline_days: int = 14):
    """Calculate baseline stats for a metric"""
    if len(values) < 3:
        return None
    
    nums = [v["value"] for v in values if v["value"] is not None]
    if len(nums) < 3:
        return None
    
    return {
        "mean": mean(nums),
        "stdev": stdev(nums) if len(nums) > 1 else 0,
        "min": min(nums),
        "max": max(nums),
        "count": len(nums)
    }

def detect_anomalies(readings: dict, config: dict):
    """Detect anomalies based on thresholds and baselines"""
    alerts = []
    thresholds = config.get("thresholds", {})
    
    # Temperature check
    if "temperature" in readings and readings["temperature"]:
        recent = readings["temperature"][-1]
        value = recent["value"]
        
        if value and value >= thresholds.get("temperature_high_f", 100.4):
            alerts.append({
                "type": "temperature_high",
                "severity": "warning" if value < 101.5 else "critical",
                "value": value,
                "threshold": thresholds.get("temperature_high_f", 100.4),
                "message": f"🌡️ Elevated temperature: {value}°F",
                "timestamp": recent["timestamp"]
            })
    
    # Heart rate check
    if "heart_rate" in readings and readings["heart_rate"]:
        recent_hr = [r["value"] for r in readings["heart_rate"][-10:] if r["value"]]
        if recent_hr:
            avg_hr = mean(recent_hr)
            if avg_hr >= thresholds.get("heart_rate_high", 120):
                alerts.append({
                    "type": "heart_rate_high",
                    "severity": "warning",
                    "value": avg_hr,
                    "threshold": thresholds.get("heart_rate_high", 120),
                    "message": f"💓 Elevated heart rate: {avg_hr:.0f} bpm (10-reading avg)",
                    "timestamp": readings["heart_rate"][-1]["timestamp"]
                })
    
    # Sleep check (past 3 nights vs baseline)
    if "sleep_hours" in readings and len(readings["sleep_hours"]) >= 7:
        recent_3 = [r["value"] for r in readings["sleep_hours"][-3:] if r["value"]]
        baseline_sleep = [r["value"] for r in readings["sleep_hours"][:-3] if r["value"]]
        
        if recent_3 and baseline_sleep:
            recent_avg = mean(recent_3)
            baseline_avg = mean(baseline_sleep)
            
            if recent_avg < baseline_avg * 0.7:  # 30% degradation
                alerts.append({
                    "type": "sleep_degradation",
                    "severity": "info",
                    "value": recent_avg,
                    "baseline": baseline_avg,
                    "message": f"😴 Sleep degradation: {recent_avg:.1f}h avg (was {baseline_avg:.1f}h)",
                    "timestamp": datetime.now().isoformat()
                })
    
    return alerts

def generate_summary(readings: dict, config: dict):
    """Generate a human-readable health summary"""
    lines = [f"# Health Summary for {config.get('human_name', 'Human')}", ""]
    lines.append(f"Generated: {datetime.now().strftime('%Y-%m-%d %H:%M')}\n")
    
    for metric, values in readings.items():
        if not values:
            continue
        
        baseline = calculate_baseline(values)
        if baseline:
            recent = values[-1]["value"] if values else None
            lines.append(f"**{metric.replace('_', ' ').title()}**")
            lines.append(f"- Latest: {recent}")
            lines.append(f"- Average: {baseline['mean']:.1f}")
            lines.append(f"- Range: {baseline['min']:.1f} - {baseline['max']:.1f}")
            lines.append("")
    
    return "\n".join(lines)

def main():
    import argparse
    parser = argparse.ArgumentParser(description="Analyze health data")
    parser.add_argument("--days", type=int, default=7, help="Days to analyze")
    parser.add_argument("--alerts-only", action="store_true", help="Only show alerts")
    args = parser.parse_args()
    
    config = load_config()
    data_dir = Path(config.get("data_dir", "./data"))
    
    readings = load_readings(data_dir, args.days)
    
    if not readings:
        print("No readings found. Run import_health.py first.")
        return
    
    alerts = detect_anomalies(readings, config)
    
    if alerts:
        print("## Alerts\n")
        for alert in alerts:
            severity_icon = "🔴" if alert["severity"] == "critical" else "🟡" if alert["severity"] == "warning" else "🔵"
            print(f"{severity_icon} {alert['message']}")
        print()
    
    if not args.alerts_only:
        summary = generate_summary(readings, config)
        print(summary)

if __name__ == "__main__":
    main()
