#!/usr/bin/env python3
"""
Deep Research Tool for Clawdbot
Performs iterative web searches via local SearXNG and synthesizes comprehensive reports.
Privacy-first: No Google, routes through VPN.
"""

import asyncio
import aiohttp
import json
import sys
import re
import unicodedata
from urllib.parse import urlencode, quote_plus
from bs4 import BeautifulSoup
from typing import Optional, List, Dict, Any
from datetime import datetime

# Configuration
SEARXNG_URL = "http://localhost:8888"
MAX_ITERATIONS = 5
RESULTS_PER_PAGE = 10
MIN_RESULTS_THRESHOLD = 3
PAGE_CONTENT_LIMIT = 2000  # words
REQUEST_TIMEOUT = 20
MAX_RETRIES = 2

IGNORED_DOMAINS = [
    "youtube.com", "facebook.com", "twitter.com", "instagram.com",
    "tiktok.com", "pinterest.com", "linkedin.com"
]

QUERY_REFINEMENTS = [
    "detailed analysis",
    "comprehensive guide",
    "in-depth review",
    "research findings",
    "technical overview"
]


class DeepResearch:
    def __init__(self, query: str, max_iterations: int = MAX_ITERATIONS):
        self.original_query = query
        self.current_query = query
        self.max_iterations = max_iterations
        self.seen_urls: set = set()
        self.results: List[Dict[str, Any]] = []
        self.citations: List[Dict[str, str]] = []

    async def search(self, query: str, offset: int = 0) -> List[Dict]:
        """Search SearXNG and return results."""
        params = {
            "q": query,
            "format": "json",
            "pageno": (offset // RESULTS_PER_PAGE) + 1
        }

        url = f"{SEARXNG_URL}/search?{urlencode(params)}"

        try:
            async with aiohttp.ClientSession() as session:
                async with session.get(url, timeout=aiohttp.ClientTimeout(total=REQUEST_TIMEOUT)) as resp:
                    if resp.status == 200:
                        data = await resp.json()
                        return data.get("results", [])
        except Exception as e:
            print(f"[!] Search error: {e}", file=sys.stderr)

        return []

    def is_valid_url(self, url: str) -> bool:
        """Check if URL should be processed."""
        if not url or url in self.seen_urls:
            return False

        for domain in IGNORED_DOMAINS:
            if domain in url.lower():
                return False

        return True

    async def fetch_content(self, url: str) -> Optional[str]:
        """Fetch and extract readable content from URL."""
        for attempt in range(MAX_RETRIES):
            try:
                headers = {
                    "User-Agent": "Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36"
                }

                async with aiohttp.ClientSession() as session:
                    async with session.get(url, headers=headers, timeout=aiohttp.ClientTimeout(total=REQUEST_TIMEOUT)) as resp:
                        if resp.status != 200:
                            continue

                        html = await resp.text()
                        soup = BeautifulSoup(html, "html.parser")

                        # Remove script and style elements
                        for tag in soup(["script", "style", "nav", "header", "footer", "aside"]):
                            tag.decompose()

                        # Extract text
                        text = soup.get_text(separator=" ", strip=True)

                        # Normalize
                        text = unicodedata.normalize("NFKC", text)
                        text = re.sub(r"\s+", " ", text).strip()

                        # Limit words
                        words = text.split()[:PAGE_CONTENT_LIMIT]
                        content = " ".join(words)

                        if len(content) > 100:
                            return content

            except Exception as e:
                if attempt < MAX_RETRIES - 1:
                    await asyncio.sleep(1)
                    continue

        return None

    def refine_query(self, iteration: int) -> str:
        """Refine query for next iteration."""
        if iteration < len(QUERY_REFINEMENTS):
            return f"{self.original_query} {QUERY_REFINEMENTS[iteration]}"
        return self.original_query

    async def process_result(self, result: Dict) -> Optional[Dict]:
        """Process a single search result."""
        url = result.get("url", "")
        title = result.get("title", "")
        snippet = result.get("content", "")

        if not self.is_valid_url(url):
            return None

        self.seen_urls.add(url)

        # Try to fetch full content
        content = await self.fetch_content(url)

        if not content:
            # Fall back to snippet
            content = snippet if len(snippet) > 50 else None

        if content:
            return {
                "title": title,
                "url": url,
                "content": content,
                "snippet": snippet
            }

        return None

    async def run(self) -> str:
        """Execute deep research and return markdown report."""
        print(f"[*] Starting deep research: {self.original_query}", file=sys.stderr)

        for iteration in range(self.max_iterations):
            query = self.refine_query(iteration) if iteration > 0 else self.original_query
            offset = iteration * RESULTS_PER_PAGE

            print(f"[*] Iteration {iteration + 1}/{self.max_iterations}: {query[:50]}...", file=sys.stderr)

            search_results = await self.search(query, offset)

            if not search_results:
                print(f"[!] No results for iteration {iteration + 1}", file=sys.stderr)
                continue

            # Process results concurrently
            tasks = [self.process_result(r) for r in search_results[:RESULTS_PER_PAGE]]
            processed = await asyncio.gather(*tasks)

            new_results = [r for r in processed if r is not None]
            self.results.extend(new_results)

            print(f"[+] Found {len(new_results)} valid results", file=sys.stderr)

            # Check if we have enough results
            if len(self.results) >= 10:
                print(f"[*] Sufficient results collected", file=sys.stderr)
                break

            # If too few results, query will be refined in next iteration
            if len(new_results) < MIN_RESULTS_THRESHOLD:
                print(f"[*] Few results, refining query...", file=sys.stderr)

        return self.generate_report()

    def generate_report(self) -> str:
        """Generate markdown research report."""
        report = []

        # Header
        report.append(f"# Deep Research Report")
        report.append(f"\n**Query:** {self.original_query}")
        report.append(f"**Date:** {datetime.now().strftime('%Y-%m-%d %H:%M')}")
        report.append(f"**Sources:** {len(self.results)}")
        report.append("\n---\n")

        # Summary section
        report.append("## Research Findings\n")

        if not self.results:
            report.append("*No results found. Try refining your query.*\n")
            return "\n".join(report)

        # Content from each source
        for i, result in enumerate(self.results, 1):
            title = result["title"]
            url = result["url"]
            content = result["content"]

            # Truncate content for display
            preview = content[:500] + "..." if len(content) > 500 else content

            report.append(f"### [{i}] {title}\n")
            report.append(f"**Source:** {url}\n")
            report.append(f"\n{preview}\n")

            self.citations.append({"num": i, "title": title, "url": url})

        # Citations section
        report.append("\n---\n")
        report.append("## Sources\n")
        for cite in self.citations:
            report.append(f"{cite['num']}. [{cite['title']}]({cite['url']})")

        return "\n".join(report)


async def main():
    if len(sys.argv) < 2:
        print("Usage: deep_research.py 'your research question'", file=sys.stderr)
        sys.exit(1)

    query = " ".join(sys.argv[1:])
    researcher = DeepResearch(query)
    report = await researcher.run()
    print(report)


if __name__ == "__main__":
    asyncio.run(main())
