---
name: prediction-market-arbitrageur
description: Meta-skill for orchestrating topic-monitor, polymarket-odds, and simmer-weather to detect potential news-vs-market mispricing in prediction markets. Use when users want a clear, step-by-step LM workflow for monitoring breaking signals, reading current Polymarket probabilities, computing confidence/price deltas, and producing alert-first arbitrage decisions.
homepage: https://clawhub.ai
user-invocable: true
disable-model-invocation: false
metadata: {"openclaw":{"emoji":"chart_with_upwards_trend","requires":{"bins":["python3","node","npx"],"env":["SIMMER_API_KEY"],"config":[]},"note":"Requires local installation of topic-monitor, polymarket-odds, and simmer-weather via ClawHub."}}
---

# Purpose

Use this meta-skill to coordinate three existing ClawHub skills into one causal arbitrage workflow:

1. Detect new high-signal news about a target event.
2. Fetch current market-implied probability from Polymarket.
3. Compare `news confidence` vs `market probability`.
4. Emit actionable alert, optionally followed by explicit execution guidance.

This skill does not replace the underlying skills. It defines how to combine them correctly.

# Required Installed Skills

This meta-skill assumes these are already installed locally:

- `topic-monitor` (inspected: latest `1.3.4`)
- `polymarket-odds` (inspected: latest `1.0.0`)
- `simmer-weather` (inspected: latest `1.7.1`, execution proxy pattern)

Install/refresh with ClawHub:

```bash
npx -y clawhub@latest install topic-monitor
npx -y clawhub@latest install polymarket-odds
npx -y clawhub@latest install simmer-weather
npx -y clawhub@latest update --all
```

Verify:

```bash
npx -y clawhub@latest list
python3 skills/topic-monitor/scripts/monitor.py --help
node skills/polymarket-odds/polymarket.mjs --help
python3 skills/simmer-weather/weather_trader.py --help
```

If any command fails, stop and report missing dependency or wrong install path.

# Inputs the LM Must Collect First

- `ceo_name`
- `company_name`
- `event_hypothesis` (for example: `CEO X resigns within 30 days`)
- `market_query` (for polymarket search)
- `topic_id` (stable ID in `topic-monitor`)
- `monitor_interval_minutes` (default: `5`)
- `min_news_confidence` (default: `0.80`)
- `min_delta` (default: `0.25`)
- `execution_mode` (`alert-only` or `execution-plan`)

Do not continue with implicit trading assumptions if these are missing.

# Skill Responsibilities (What Each Tool Actually Does)

## `topic-monitor`

Use for continuous signal discovery and scoring.

Operationally relevant behavior:
- Topic config via `scripts/manage_topics.py`.
- Monitoring loop via `scripts/monitor.py`.
- Priority/score generated by its scoring logic.
- Alert queue retrieval via `scripts/process_alerts.py --json`.

This is the source of `news confidence` candidates.

## `polymarket-odds`

Use for live market probability lookups.

Operationally relevant behavior:
- `search <query>` to find matching events/markets.
- `market <slug>` to inspect specific market pricing.
- Outputs percentage-formatted odds that must be normalized to `[0,1]`.

This is the source of `market probability`.

## `simmer-weather`

Primary design is weather strategy, but in this chain it is treated as execution proxy reference because it uses Simmer SDK trade endpoints and live/dry-run safety pattern.

Operationally relevant behavior:
- Requires `SIMMER_API_KEY`.
- Supports dry-run and live execution modes.
- Demonstrates guarded trading workflow and position checks.

In this meta-skill, it is not the signal engine. It is the execution pattern reference.

# Canonical Causal Chain

Use this exact chain:

1. `topic-monitor` heartbeat every 5 minutes.
2. Match target rumor pattern (`resignation`, `ceo_name`, `company_name`).
3. Accept only high-confidence signal (`news_confidence >= 0.80`).
4. Query `polymarket-odds` for matching market and read current yes probability.
5. Compute `delta = news_confidence - market_probability`.
6. If `delta >= min_delta`, trigger arbitrage alert.
7. If `execution_mode=execution-plan`, output explicit next trading step; do not auto-trade unless user explicitly asks.

# Data Contract Between Skills

Normalize all values into one record before decisioning:

```json
{
  "topic_id": "ceo-resignation-acme",
  "event_hypothesis": "CEO X resigns",
  "news_confidence": 0.82,
  "news_signal_time": "2026-02-14T14:05:00Z",
  "market_slug": "will-ceo-x-resign",
  "market_probability": 0.40,
  "market_snapshot_time": "2026-02-14T14:06:00Z",
  "delta": 0.42,
  "decision": "buy_yes_candidate"
}
```

Hard rules:
- Reject stale signal if `news_signal_time` is older than 30 minutes.
- Reject stale market snapshot older than 5 minutes.
- Never compare percentages and decimals mixed. Convert all to decimals first.

# LM Execution Playbook

## Step A: Configure topic once

```bash
python3 skills/topic-monitor/scripts/manage_topics.py add \
  "CEO Resignation - <company_name>" \
  --id <topic_id> \
  --query "<ceo_name> resignation <company_name> CEO stepping down" \
  --keywords "resignation,<ceo_name>,<company_name>,CEO,board,step down" \
  --frequency hourly \
  --importance high \
  --channels telegram \
  --context "Prediction market mispricing detection"
```

## Step B: Run heartbeat loop externally (every 5 min)

```bash
python3 skills/topic-monitor/scripts/monitor.py --topic <topic_id> --force
python3 skills/topic-monitor/scripts/process_alerts.py --json
```

Use max recent score for confidence extraction.

## Step C: Pull market probability

```bash
node skills/polymarket-odds/polymarket.mjs search "<market_query>"
node skills/polymarket-odds/polymarket.mjs market <market_slug>
```

Extract yes-price and normalize (`40% -> 0.40`).

## Step D: Decide

Formula:
- `delta = news_confidence - market_probability`
- Trigger if `news_confidence >= min_news_confidence` and `delta >= min_delta`

## Step E: Emit output

If triggered, emit:

`🚨 ARBITRAGE: News bestätigen, Markt schläft. Kauf empfohlen.`

Plus structured fields:
- `news_confidence`
- `market_probability`
- `delta`
- `signal_age_minutes`
- `market_age_minutes`
- `recommendation`

# Output Modes

## `alert-only`

Return recommendation and confidence math only. No execution step.

## `execution-plan`

Return recommendation plus explicit manual next actions using installed `simmer-weather` runtime pattern:
- check API key present
- run dry-run first
- require explicit user confirmation before any live action

# Guardrails for the LM

- Do not fabricate market slugs or prices.
- Do not promote execution when confidence math is below threshold.
- Do not issue live-trade instructions without clear user opt-in.
- Surface uncertainty explicitly when matching query to market is ambiguous.
- Prefer false-negative over false-positive when news credibility is weak.

# Failure Handling

- Missing skill install: output exact missing path/command.
- Missing env var (`SIMMER_API_KEY`): degrade to `alert-only`.
- No market match: return `no_trade` with retry query suggestions.
- Conflicting signals: require two independent high-confidence hits before alerting.

# Why This Meta-Skill Exists

Without orchestration, each tool solves only a fragment:
- `topic-monitor` detects events but has no market-price context.
- `polymarket-odds` shows prices but no external signal confidence.
- `simmer-weather` demonstrates execution mechanics but is not a generic event detector.

This meta-skill binds those fragments into one coherent arbitrage decision process that an LM can execute consistently.
