# Praesidia OpenClaw Skill

**Agent Identity, Verification & Guardrails for OpenClaw**

This skill enables OpenClaw AI assistants to verify AI agent identities, check trust scores, discover agents, and apply security guardrails through Praesidia's trust and verification layer.

---

## Overview

Praesidia provides a trust framework for AI agents, similar to how SSL certificates work for websites. This OpenClaw skill allows AI assistants to:

- ✅ **Verify agent identity** - Check if an agent is registered and verified
- 📊 **Check trust scores** - View 0-100 trust ratings with detailed breakdowns
- 🔍 **Discover agents** - Search marketplace for public agents by capability
- 📋 **Fetch agent cards** - Get A2A-compliant agent metadata
- 🛡️ **Make trust decisions** - Help users decide if an agent is safe to use
- 🔒 **Apply guardrails** - Add security policies, content moderation, and compliance rules

---

## How It Works

### For AI Assistants (OpenClaw)

When a user asks about agent safety, wants to find agents, or needs to secure their agents, this skill provides clear guidance on:

1. **Which API endpoints to call** (with exact URL formats)
2. **How to authenticate** (using Bearer tokens)
3. **What data to present** (trust scores, capabilities, compliance, guardrails)
4. **How to explain trust** (user-friendly descriptions of levels)
5. **How to apply guardrails** (security policies, content filters, compliance rules)

The skill uses `web_fetch` tool calls to interact with Praesidia's REST API.

### For Users

Users can ask natural language questions like:
- "Is agent chatbot-v2 safe?"
- "Find me a data analysis agent"
- "Show me all my registered agents"
- "What's the trust score for agent xyz?"
- "What guardrails are configured for my agent?"
- "Add PII detection to my chatbot"
- "Apply security policies to protect my agent"
- "Check if this message passes guardrails"

The AI assistant uses this skill to provide accurate, helpful answers.

---

## Installation

### From ClawHub (Recommended)

```bash
clawhub install praesidia
```

### Manual Installation

1. Copy this folder to: `~/.openclaw/skills/praesidia/`
2. Restart OpenClaw to pick up the skill

---

## Configuration

Add your Praesidia API key to `~/.openclaw/openclaw.json`:

```json
{
  "skills": {
    "entries": {
      "praesidia": {
        "apiKey": "pk_live_your_key_here",
        "env": {
          "PRAESIDIA_API_URL": "https://api.praesidia.ai"
        }
      }
    }
  }
}
```

### Get an API Key

1. Sign up at [https://praesidia.ai](https://praesidia.ai)
2. Navigate to Settings → API Keys
3. Create a new API key
4. Copy and paste into OpenClaw config

### Environment Options

- **Production:** `https://api.praesidia.ai` (default)
- **Local dev:** `http://localhost:5001` or `http://localhost:3000`
- **Custom:** Your own Praesidia deployment URL

---

## For AI Assistants: Understanding This Skill

### Core Concept: Trust Scores

Every agent in Praesidia has a trust score (0-100) and a trust level:

| Score | Level | Meaning | Action |
|-------|-------|---------|--------|
| 90-100 | VERIFIED | Fully vetted, verified identity | ✅ Recommend |
| 70-89 | STANDARD | Good reputation, basic checks | ✅ Generally safe |
| 50-69 | LIMITED | Minimal verification | ⚠️ Caution |
| 0-49 | UNTRUSTED | Not verified or bad reputation | ❌ Warn user |

**Always present both:** "Trust Score: 92.5/100 (VERIFIED)"

### Key API Endpoints

#### 1. Verify an Agent (Trust & Identity)

```javascript
GET ${PRAESIDIA_API_URL}/agents/{agentId}/agent-card
Authorization: Bearer ${PRAESIDIA_API_KEY}
```

**Returns:** Complete agent card with trust data, capabilities, compliance info.

**Use when:** User asks "Is agent X safe?" or "Verify agent X"

#### 2. Discover Agents

```javascript
GET ${PRAESIDIA_API_URL}/agents/discovery?visibility=PUBLIC&search=keyword
```

**Returns:** List of matching agents with trust scores.

**Use when:** User asks "Find agents that can X" or "Show me chatbot agents"

**Query params:**
- `visibility=PUBLIC` - Public marketplace agents
- `role=SERVER` - Service providers (most common)
- `role=CLIENT` - Service consumers
- `status=ACTIVE` - Only active agents
- `search=keyword` - Search by name/description

#### 3. List User's Agents

```javascript
GET ${PRAESIDIA_API_URL}/agents/discovery
Authorization: Bearer ${PRAESIDIA_API_KEY}
```

**Returns:** All agents user has access to (own + team + org).

**Use when:** User asks "Show my agents" or "List my registered agents"

#### 4. List Guardrails for an Agent

```javascript
GET ${PRAESIDIA_API_URL}/organizations/{orgId}/guardrails?agentId={agentId}
Authorization: Bearer ${PRAESIDIA_API_KEY}
```

**Returns:** List of configured guardrails with type, category, action, scope, and trigger stats.

**Use when:** User asks "What guardrails are configured?" or "Show security policies"

#### 5. Get Guardrail Templates

```javascript
GET ${PRAESIDIA_API_URL}/organizations/{orgId}/guardrails/templates
Authorization: Bearer ${PRAESIDIA_API_KEY}
```

**Returns:** Available guardrail templates by category (Content, Security, Compliance, Brand, Accuracy).

**Use when:** User asks "What guardrails are available?" or "Show me security templates"

#### 6. Apply a Guardrail

```javascript
POST ${PRAESIDIA_API_URL}/organizations/{orgId}/guardrails
Authorization: Bearer ${PRAESIDIA_API_KEY}
Content-Type: application/json

{
  "name": "PII Detection",
  "agentId": "agent-id",
  "template": "PII_DETECTION",
  "type": "ML",
  "category": "SECURITY",
  "scope": "BOTH",
  "action": "REDACT",
  "severity": "HIGH",
  "isEnabled": true
}
```

**Returns:** Created guardrail configuration.

**Use when:** User asks "Add PII detection" or "Apply toxic language filter"

#### 7. Validate Content Against Guardrails

```javascript
POST ${PRAESIDIA_API_URL}/organizations/{orgId}/guardrails/validate
Authorization: Bearer ${PRAESIDIA_API_KEY}
Content-Type: application/json

{
  "content": "Text to validate",
  "agentId": "agent-id",
  "scope": "INPUT"
}
```

**Returns:** Pass/fail status, triggered guardrails, suggested actions, modified content.

**Use when:** User asks "Check if this message passes guardrails"

### Response Format Guidelines

When presenting agent information, always include:

1. **Trust indicator** - Visual indicator (✅/⚠️/❌) + score
2. **Status** - ACTIVE/INACTIVE
3. **Capabilities** - What the agent can do
4. **Compliance** - SOC2, GDPR, HIPAA (if any)
5. **Action recommendation** - What user should do

**Good example:**
```
✅ ChatBot V2 is verified and safe to use!

Trust Score: 92.5/100 (VERIFIED)
Status: ACTIVE
Capabilities: message:send, task:create, data:analyze
Compliance: SOC2, GDPR
Last verified: 2 days ago

Recommendation: This agent is fully verified and safe for production use.

Agent card: https://api.praesidia.ai/agents/chatbot-v2/agent-card
```

**Bad example:**
```
Agent is verified.
```

### Error Handling

When API calls fail, explain clearly:

| Error | User-Friendly Message |
|-------|----------------------|
| 401 | "Your API key is invalid. Check ~/.openclaw/openclaw.json" |
| 403 | "You don't have permission to access this agent" |
| 404 | "Agent not found. Double-check the agent ID" |
| 500 | "Praesidia API is temporarily unavailable. Try again in a moment" |

### Guardrails Framework

Guardrails are security policies that protect agents from harmful content, security threats, and compliance violations.

#### Guardrail Types

| Type | Description | Use Case |
|------|-------------|----------|
| **RULE** | Regex/keyword matching | Fast, simple pattern detection |
| **ML** | Machine learning model | Balanced accuracy and speed |
| **LLM** | LLM-powered validation | Most accurate, slower |

#### Guardrail Categories

1. **CONTENT** - Content moderation (toxic language, hate speech, profanity)
2. **SECURITY** - Security checks (PII, API keys, prompt injection, jailbreaks)
3. **COMPLIANCE** - Regulatory compliance (GDPR, HIPAA, financial/medical advice)
4. **BRAND** - Brand safety (competitor mentions, tone, voice, off-topic)
5. **ACCURACY** - Accuracy checks (hallucinations, fact-checking, consistency)

#### Guardrail Actions

| Action | Behavior | Use Case |
|--------|----------|----------|
| **BLOCK** | Stop request/response entirely | Critical security violations |
| **WARN** | Log but allow through | Compliance monitoring |
| **REDACT** | Mask offending content | Sensitive data (PII, credentials) |
| **REPLACE** | Substitute alternative | Brand voice, tone correction |
| **RETRY** | Retry with modified prompt | Hallucination mitigation |
| **ESCALATE** | Send to human review | Edge cases, uncertain violations |

#### Guardrail Scope

- **INPUT** - Validate user input only
- **OUTPUT** - Validate agent output only
- **BOTH** - Validate both directions (recommended for most)

#### Available Templates

**Content Moderation:**
- TOXIC_LANGUAGE, PROFANITY_FILTER, HATE_SPEECH, VIOLENCE_DETECTION, ADULT_CONTENT

**Security:**
- PII_DETECTION, CREDIT_CARD_DETECTION, SSN_DETECTION, API_KEY_DETECTION, PROMPT_INJECTION, JAILBREAK_DETECTION

**Compliance:**
- FINANCIAL_ADVICE, MEDICAL_ADVICE, LEGAL_ADVICE, GDPR_COMPLIANCE, HIPAA_COMPLIANCE

**Brand Safety:**
- COMPETITOR_MENTIONS, POSITIVE_TONE, BRAND_VOICE, OFF_TOPIC_DETECTION

**Accuracy:**
- HALLUCINATION_DETECTION, FACT_CHECKING, SOURCE_VALIDATION, CONSISTENCY_CHECK

---

### Common User Patterns

#### Pattern 1: Safety Check
```
User: "Is agent xyz safe?"
You should:
1. Fetch agent card
2. Check trust score
3. Explain what the score means
4. Make a recommendation
```

#### Pattern 2: Agent Discovery
```
User: "Find me an agent that can analyze spreadsheets"
You should:
1. Search discovery API with keyword "spreadsheet"
2. Filter by role=SERVER (agents that provide services)
3. Sort by trust score (highest first)
4. Present top 3-5 results with trust indicators
```

#### Pattern 3: Fleet Management
```
User: "Show all my inactive agents"
You should:
1. Fetch user's agents with status=INACTIVE
2. List them with trust scores
3. Suggest actions (activate, review, delete)
```

---

## Use Cases

### 1. Agent Safety Verification

**Scenario:** User wants to integrate with a third-party agent but isn't sure if it's safe.

**Solution:** Check trust score, review compliance certifications, verify capabilities match expectations.

### 2. Agent Marketplace Discovery

**Scenario:** User needs an agent for a specific task (e.g., data analysis, file processing).

**Solution:** Search public agents, filter by capability, rank by trust score.

### 3. Compliance Requirements

**Scenario:** Enterprise user needs agents with SOC2 or GDPR compliance.

**Solution:** Verify agent card includes required compliance certifications.

### 4. Fleet Management

**Scenario:** Organization has multiple agents and needs visibility into their status and trust.

**Solution:** List all organization agents with trust scores and statuses.

### 5. Security Hardening

**Scenario:** User needs to protect their agent from security threats like prompt injection, PII leaks, and jailbreak attempts.

**Solution:** Apply layered security guardrails (PII_DETECTION + PROMPT_INJECTION + JAILBREAK_DETECTION) with appropriate actions.

### 6. Compliance Enforcement

**Scenario:** Enterprise needs to ensure agents comply with regulations like GDPR, HIPAA, or SOC2.

**Solution:** Apply compliance guardrails, enable audit logging, validate content before deployment.

### 7. Brand Protection

**Scenario:** Company wants agents to maintain brand voice and avoid mentioning competitors.

**Solution:** Apply brand safety guardrails (POSITIVE_TONE + COMPETITOR_MENTIONS + BRAND_VOICE) with REPLACE or WARN actions.

---

## Architecture Context

### How Praesidia Fits Into Agent Ecosystem

```
┌─────────────────────────────────────────────────────────┐
│                    OpenClaw User                         │
│  "Is agent xyz safe? Find me a chatbot agent."         │
└──────────────────────┬──────────────────────────────────┘
                       │
                       ▼
┌─────────────────────────────────────────────────────────┐
│                  OpenClaw AI Assistant                   │
│  Uses: praesidia skill (this skill)                     │
└──────────────────────┬──────────────────────────────────┘
                       │
                       │ web_fetch() calls
                       ▼
┌─────────────────────────────────────────────────────────┐
│                Praesidia REST API                        │
│  - Agent verification                                    │
│  - Trust score calculation                               │
│  - Agent discovery                                       │
└──────────────────────┬──────────────────────────────────┘
                       │
                       ▼
┌─────────────────────────────────────────────────────────┐
│              Praesidia Backend (be-core)                 │
│  - Database of registered agents                         │
│  - Trust calculation engine                              │
│  - Compliance tracking                                   │
└─────────────────────────────────────────────────────────┘
```

### Guardrails in the Ecosystem

```
┌──────────────────────────────────────────────────────────┐
│                     User Input                            │
└───────────────────────┬──────────────────────────────────┘
                        │
                        ▼
┌──────────────────────────────────────────────────────────┐
│              INPUT Guardrails (Applied)                   │
│  • PII Detection (REDACT)                                │
│  • Prompt Injection (BLOCK)                              │
│  • Toxic Language (BLOCK)                                │
└───────────────────────┬──────────────────────────────────┘
                        │
                        ▼
┌──────────────────────────────────────────────────────────┐
│                 Agent Processing                          │
└───────────────────────┬──────────────────────────────────┘
                        │
                        ▼
┌──────────────────────────────────────────────────────────┐
│             OUTPUT Guardrails (Applied)                   │
│  • Hallucination Detection (RETRY)                       │
│  • Compliance Check (WARN)                               │
│  • Brand Voice (REPLACE)                                 │
└───────────────────────┬──────────────────────────────────┘
                        │
                        ▼
┌──────────────────────────────────────────────────────────┐
│                  Safe Output to User                      │
└──────────────────────────────────────────────────────────┘
```

### What Gets Verified

Praesidia verifies multiple aspects of agent identity:

1. **Identity Verification**
   - Owner verification (organization, team, individual)
   - Domain ownership (for web-hosted agents)
   - Credential validation

2. **Behavioral Trust**
   - Interaction history
   - User feedback
   - Incident reports
   - Response reliability

3. **Security Posture**
   - Authentication methods
   - Encryption standards
   - Security audits
   - Vulnerability scanning

4. **Compliance**
   - Regulatory certifications (SOC2, GDPR, HIPAA)
   - Privacy policy
   - Terms of service
   - Data handling practices

### Trust Score Calculation

Trust score is a weighted average of components:

```
Trust Score = (
  Identity * 0.30 +
  Behavior * 0.25 +
  Security * 0.25 +
  Compliance * 0.20
)
```

Each component is scored 0-100, then combined.

---

## Agent Card Format

Agent cards follow the A2A (Agent-to-Agent) protocol standard with Praesidia extensions.

### Standard A2A Fields

```json
{
  "name": "ChatBot V2",
  "description": "A conversational AI agent",
  "version": "2.0.0",
  "url": "https://chatbot.example.com",
  "capabilities": {
    "streaming": true,
    "pushNotifications": false
  },
  "skills": [
    {
      "id": "message:send",
      "name": "Send Message",
      "description": "Send text messages"
    }
  ]
}
```

### Praesidia Extensions

```json
{
  "praesidia": {
    "agentId": "agent-uuid-123",
    "trustScore": 92.5,
    "trustLevel": "VERIFIED",
    "trustComponents": {
      "identity": 95,
      "behavior": 90,
      "security": 92,
      "compliance": 93
    },
    "verificationDate": "2026-02-01T00:00:00Z",
    "status": "ACTIVE",
    "compliance": ["SOC2", "GDPR"],
    "organizationId": "org-456"
  }
}
```

---

## API Authentication

### Using API Keys

All authenticated endpoints require a Bearer token:

```javascript
Authorization: Bearer pk_live_abc123...
```

The skill automatically uses `${PRAESIDIA_API_KEY}` from OpenClaw config.

### Public vs Private Agents

- **Public agents** - No auth required (but providing auth may return more results)
- **Private/Team/Org agents** - Auth required

---

## Developer Information

### Praesidia Platform

- **Homepage:** https://praesidia.ai
- **API Docs:** https://app.praesidia.ai/docs/api
- **A2A Protocol:** https://a2a-protocol.org
- **Discord:** https://discord.gg/e9EwZfHS
- **Support:** hello@praesidia.ai

### Related Projects

- **be-core** - Praesidia backend (NestJS, PostgreSQL)
- **ui-core** - Praesidia dashboard (Next.js, React)
- **sdk-a2a** - A2A protocol SDK with Praesidia integration
- **mcp-server-praesidia** - MCP server for Praesidia

### OpenClaw Integration

This skill is part of the OpenClaw ecosystem:

- **OpenClaw Docs:** https://openclaw.io/docs
- **Skills Format:** https://docs.clawd.bot/tools/skills
- **ClawHub:** https://clawhub.ai (skill marketplace)

---

## Testing

### Local Testing

If running Praesidia locally:

```json
{
  "skills": {
    "entries": {
      "praesidia": {
        "apiKey": "pk_test_local",
        "env": {
          "PRAESIDIA_API_URL": "http://localhost:5001"
        }
      }
    }
  }
}
```

### Test Queries

Ask OpenClaw:

```
"List my agents"
"Find public chatbot agents"
"Is agent test-123 safe?"
"Show me agents with SOC2 compliance"
```

Expected: AI should make appropriate API calls and present formatted results.

---

## Troubleshooting

### Skill Not Loading

1. Check file location: `~/.openclaw/skills/praesidia/SKILL.md`
2. Verify YAML frontmatter is valid
3. Restart OpenClaw
4. Check OpenClaw logs for errors

### Authentication Errors

1. Verify API key in `~/.openclaw/openclaw.json`
2. Check key format: `pk_live_...` or `pk_test_...`
3. Ensure key has not been revoked
4. Test key directly with curl:
   ```bash
   curl -H "Authorization: Bearer pk_live_..." \
        https://api.praesidia.ai/agents/discovery
   ```

### No Results Found

1. Check if agent exists in Praesidia
2. Verify user has access (for private agents)
3. Try with `visibility=PUBLIC` filter
4. Check spelling of agent ID

---

## Security & Privacy

### Data Handling

- **API keys** - Stored locally in OpenClaw config, never sent to users
- **Agent data** - Fetched in real-time from Praesidia API
- **No caching** - Fresh data on every request (OpenClaw may cache)
- **HTTPS** - All production requests use TLS

### Privacy

- Public agent data is visible to everyone
- Private agent data requires authentication
- Trust scores are calculated server-side
- No user data is logged by this skill

---

## Contributing

### Improving This Skill

1. Test with various agent types
2. Suggest better response formats
3. Add more examples to SKILL.md
4. Report issues or edge cases
5. Share usage patterns

### Related Skills

Consider also installing:

- **praesidia-a2a** - OAuth and A2A protocol implementation
- **mcp-praesidia** - MCP server integration

---

## Changelog

### v1.0.0 (Initial Release)
- Agent verification with trust scores
- Agent discovery and search
- User agent listing
- Trust level explanations
- Error handling and user guidance

---

## License

MIT License - See main Praesidia repository

---

## Quick Start for AI Assistants

**tl;dr:** When user asks about agent safety, wants to find agents, or needs to secure their agents:

### For Verification & Discovery:

1. Use `web_fetch()` to call Praesidia API
2. Always include trust score in results (e.g., "92.5/100 (VERIFIED)")
3. Explain what trust level means
4. Recommend action based on trust
5. Present capabilities and compliance clearly

**Key URLs:**
- Verify: `${PRAESIDIA_API_URL}/agents/{id}/agent-card`
- Discover: `${PRAESIDIA_API_URL}/agents/discovery`
- Auth: `Authorization: Bearer ${PRAESIDIA_API_KEY}`

### For Guardrails:

1. List available templates by category
2. Recommend appropriate guardrails for use case
3. Apply with proper type, scope, and action
4. Explain what each guardrail does
5. Show how to monitor trigger stats

**Key URLs:**
- List: `${PRAESIDIA_API_URL}/organizations/{orgId}/guardrails`
- Templates: `${PRAESIDIA_API_URL}/organizations/{orgId}/guardrails/templates`
- Apply: `POST ${PRAESIDIA_API_URL}/organizations/{orgId}/guardrails`
- Validate: `POST ${PRAESIDIA_API_URL}/organizations/{orgId}/guardrails/validate`

**Goals:** 
1. Help users make informed decisions about which agents to trust and use
2. Help users secure their agents with appropriate guardrails
3. Ensure compliance and safety in agent interactions
