---
name: PowerShell
description: Avoid common PowerShell mistakes — output behavior, array traps, and comparison operator gotchas.
metadata: {"clawdbot":{"emoji":"🔵","requires":{"bins":["pwsh"]},"os":["linux","darwin","win32"]}}
---

## Output Behavior
- Everything not captured goes to output — even without `return` or `Write-Output`
- `return` doesn't stop output — previous uncaptured expressions still output
- `Write-Host` bypasses pipeline — use for display only, not data
- Assign to `$null` to suppress — `$null = SomeFunction`
- `[void]` cast also suppresses — `[void](SomeFunction)`

## Array Gotchas
- Single item result is scalar, not array — `@(Get-Item .)` forces array
- Empty result is `$null`, not empty array — check with `if ($result)` carefully
- Array unrolling in pipeline — `@(1,2,3) | ForEach` sends items one by one
- `+=` on array creates new array — slow in loops, use `[System.Collections.ArrayList]`
- `,` is array operator — `,$item` wraps single item in array

## Comparison Operators
- `-eq`, `-ne`, `-gt`, `-lt` — not `==`, `!=`, `>`, `<`
- `-like` with wildcards, `-match` with regex — both return bool
- `-contains` for array membership — `$arr -contains $item`, not `$item -in $arr` (though `-in` works too)
- Case-insensitive by default — `-ceq`, `-cmatch` for case-sensitive
- `$null` on left side — `$null -eq $var` prevents array comparison issues

## String Handling
- Double quotes interpolate — `"Hello $name"` expands variable
- Single quotes literal — `'$name'` stays as literal text
- Subexpression for complex — `"Count: $($arr.Count)"` for properties/methods
- Here-strings for multiline — `@" ... "@` or `@' ... '@`
- Backtick escapes — `` `n `` for newline, `` `t `` for tab

## Pipeline
- `$_` or `$PSItem` is current object — same thing, `$_` more common
- `ForEach-Object` for pipeline — `foreach` statement doesn't take pipeline
- `-PipelineVariable` saves intermediate — `Get-Service -PV svc | Where ...`
- Pipeline processes one at a time — unless function doesn't support streaming

## Error Handling
- `$ErrorActionPreference` sets default — `Stop`, `Continue`, `SilentlyContinue`
- `-ErrorAction Stop` per command — makes non-terminating errors terminating
- `try/catch` only catches terminating — set `ErrorAction Stop` first
- `$?` is last command success — `$LASTEXITCODE` for native commands

## Common Mistakes
- No space before `{` in `if` — `if($x){` works but `if ($x) {` preferred
- `=` is assignment in conditions — use `-eq` for comparison
- Function return array unrolls — `return ,@($arr)` to keep array
- `Get-Content` returns lines array — `-Raw` for single string
- `Select-Object` creates new object — properties are copies, not references

## Cross-Platform
- `pwsh` is PowerShell 7+ — `powershell` is Windows PowerShell 5.1
- Paths use `/` or `\` — `Join-Path` for portable
- Environment vars: `$env:VAR` — works on all platforms
- Aliases differ across platforms — `ls`, `cat` may not exist, use full cmdlet names
