const express = require('express');
const { ethers } = require('ethers');
const axios = require('axios');
const cron = require('node-cron');
require('dotenv').config();

const app = express();
app.use(express.json());

// Constants
const BANKR_TOKEN = process.env.BANKR_TOKEN || '0x50D2280441372486BeecdD328c1854743EBaCb07';
const SCAN_PRICE = 5; // $5 USD
const FREE_HOLDER_THRESHOLD = 1000; // 1000 BANKR tokens

// RPC Providers
const providers = {
  ethereum: new ethers.JsonRpcProvider(process.env.ETHEREUM_RPC),
  base: new ethers.JsonRpcProvider(process.env.BASE_RPC),
  polygon: new ethers.JsonRpcProvider(process.env.POLYGON_RPC)
};

// ERC20 ABI (minimal)
const ERC20_ABI = [
  "function balanceOf(address) view returns (uint256)",
  "function decimals() view returns (uint8)",
  "function symbol() view returns (string)"
];

// Check if user has access (paid or holds BANKR)
async function checkAccess(wallet) {
  try {
    // Check BANKR balance on all chains
    for (const [chain, provider] of Object.entries(providers)) {
      try {
        const bankrContract = new ethers.Contract(BANKR_TOKEN, ERC20_ABI, provider);
        const balance = await bankrContract.balanceOf(wallet);
        const decimals = await bankrContract.decimals();
        const balanceFormatted = parseFloat(ethers.formatUnits(balance, decimals));
        
        if (balanceFormatted >= FREE_HOLDER_THRESHOLD) {
          return { access: true, reason: 'bankr_holder', balance: balanceFormatted, chain };
        }
      } catch (err) {
        // Token might not exist on this chain
        continue;
      }
    }
    
    return { access: false, reason: 'payment_required' };
  } catch (error) {
    console.error('Error checking access:', error);
    return { access: false, reason: 'error' };
  }
}

// Fetch portfolio data
async function fetchPortfolio(wallet, chain = 'ethereum') {
  const provider = providers[chain];
  
  try {
    // Get ETH balance
    const ethBalance = await provider.getBalance(wallet);
    const ethBalanceFormatted = parseFloat(ethers.formatEther(ethBalance));
    
    // Fetch token balances via Alchemy or Moralis API
    // This is a simplified version - in production, use proper APIs
    const tokens = await fetchTokenBalances(wallet, chain);
    
    // Calculate total value
    let totalValue = ethBalanceFormatted * await getEthPrice();
    
    // Add token values
    const breakdown = {
      eth: ethBalanceFormatted * await getEthPrice(),
      stablecoins: 0,
      defi: 0,
      memecoins: 0,
      other: 0
    };
    
    for (const token of tokens) {
      const value = token.balance * token.price;
      totalValue += value;
      
      // Categorize
      if (['USDC', 'USDT', 'DAI'].includes(token.symbol)) {
        breakdown.stablecoins += value;
      } else if (['UNI', 'AAVE', 'COMP', 'CRV'].includes(token.symbol)) {
        breakdown.defi += value;
      } else if (['SHIB', 'DOGE', 'PEPE'].includes(token.symbol)) {
        breakdown.memecoins += value;
      } else {
        breakdown.other += value;
      }
    }
    
    // Calculate risk score
    const riskScore = calculateRiskScore(breakdown, totalValue, tokens);
    
    return {
      wallet,
      chain,
      totalValue,
      breakdown,
      riskScore,
      tokens,
      topHoldings: tokens.slice(0, 5).map(t => ({
        symbol: t.symbol,
        value: t.balance * t.price,
        percentage: ((t.balance * t.price) / totalValue * 100).toFixed(2)
      }))
    };
  } catch (error) {
    console.error('Error fetching portfolio:', error);
    throw error;
  }
}

// Calculate risk score
function calculateRiskScore(breakdown, totalValue, tokens) {
  let score = 0;
  
  // Concentration risk (30%)
  const topHoldingPercent = tokens[0] ? (tokens[0].balance * tokens[0].price) / totalValue : 0;
  const concentrationRisk = topHoldingPercent * 100;
  score += concentrationRisk * 0.3;
  
  // Volatility risk (30%) - based on asset mix
  const memePercent = breakdown.memecoins / totalValue;
  const volatilityRisk = memePercent * 100;
  score += volatilityRisk * 0.3;
  
  // Stablecoin buffer (20%) - inverse
  const stablePercent = breakdown.stablecoins / totalValue;
  const liquidityRisk = (1 - stablePercent) * 100;
  score += liquidityRisk * 0.2;
  
  // Diversification (20%)
  const diversificationRisk = tokens.length < 5 ? 100 : Math.max(0, 100 - tokens.length * 5);
  score += diversificationRisk * 0.2;
  
  return Math.min(100, Math.round(score));
}

// Generate recommendations
function generateRecommendations(portfolio) {
  const recommendations = [];
  const { breakdown, totalValue, riskScore } = portfolio;
  
  // High memecoin exposure
  if (breakdown.memecoins / totalValue > 0.3) {
    recommendations.push("Reduce memecoin exposure from " + 
      (breakdown.memecoins / totalValue * 100).toFixed(0) + "% to 15% or less");
  }
  
  // Low stablecoin buffer
  if (breakdown.stablecoins / totalValue < 0.1) {
    recommendations.push("Add 10-20% stablecoin buffer for liquidity and stability");
  }
  
  // High concentration
  const topPercent = parseFloat(portfolio.topHoldings[0]?.percentage || 0);
  if (topPercent > 50) {
    recommendations.push("Diversify: " + portfolio.topHoldings[0].symbol + 
      " is " + topPercent + "% of portfolio (target: <30%)");
  }
  
  // High overall risk
  if (riskScore > 70) {
    recommendations.push("Portfolio is high risk. Consider rebalancing to more stable assets");
  }
  
  if (recommendations.length === 0) {
    recommendations.push("Portfolio looks well-balanced! Keep monitoring.");
  }
  
  return recommendations;
}

// Stub functions (implement with real APIs)
async function fetchTokenBalances(wallet, chain) {
  // TODO: Implement with Alchemy, Moralis, or similar
  return [];
}

async function getEthPrice() {
  try {
    const response = await axios.get('https://api.coingecko.com/api/v3/simple/price?ids=ethereum&vs_currencies=usd');
    return response.data.ethereum.usd;
  } catch (error) {
    console.error('Error fetching ETH price:', error);
    return 3000; // Fallback
  }
}

// API Routes

app.post('/api/analyze', async (req, res) => {
  try {
    const { wallet, payment_tx, chain = 'ethereum' } = req.body;
    
    if (!wallet || !ethers.isAddress(wallet)) {
      return res.status(400).json({ error: 'Invalid wallet address' });
    }
    
    // Check access
    const access = await checkAccess(wallet);
    
    if (!access.access) {
      return res.status(402).json({
        error: 'Payment required',
        message: 'Send $5 USDC or hold 1000+ BANKR tokens for free access',
        bankr_balance: access.balance || 0
      });
    }
    
    // Fetch and analyze portfolio
    const portfolio = await fetchPortfolio(wallet, chain);
    const recommendations = generateRecommendations(portfolio);
    
    res.json({
      ...portfolio,
      recommendations,
      access_reason: access.reason
    });
    
  } catch (error) {
    console.error('Error in /api/analyze:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

app.get('/api/stats', async (req, res) => {
  try {
    // Return buyback stats
    res.json({
      total_scans: 0, // TODO: Track in database
      total_revenue: 0,
      bankr_bought: 0,
      holders_served: 0
    });
  } catch (error) {
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Auto-buyback cron job (runs every hour)
cron.schedule('0 * * * *', async () => {
  console.log('⏰ Running automated buyback check...');
  
  try {
    const { execSync } = require('child_process');
    execSync('./scripts/execute-buyback.sh 100', { stdio: 'inherit' });
  } catch (error) {
    console.error('Buyback failed:', error);
  }
});

// Start server
const PORT = process.env.PORT || 3000;
app.listen(PORT, () => {
  console.log(`🚀 Portfolio Risk Analyzer API running on port ${PORT}`);
  console.log(`💎 BANKR Token: ${BANKR_TOKEN}`);
  console.log(`💰 Scan Price: $${SCAN_PRICE}`);
  console.log(`🎁 Free Access: ${FREE_HOLDER_THRESHOLD}+ BANKR`);
});
