#!/bin/bash
# Analyze crypto wallet portfolio for risk

set -e

WALLET="$1"
CHAIN="${2:-all}"

if [[ -z "$WALLET" ]]; then
  echo "Usage: $0 <wallet_address> [chain]"
  exit 1
fi

echo "🔍 Analyzing wallet: $WALLET"
echo "Chain: $CHAIN"
echo ""

# Check if user holds BANKR for free access
BANKR_BALANCE=$(node scripts/check-bankr-balance.js "$WALLET")

if (( $(echo "$BANKR_BALANCE < 1000" | bc -l) )); then
  echo "⚠️  Payment required (or hold 1000+ \$BANKR for free access)"
  echo "Current BANKR balance: $BANKR_BALANCE tokens"
  echo ""
  echo "Pay \$5 in USDC to: [Payment address]"
  exit 1
fi

echo "✅ Access granted (BANKR holder)"
echo ""

# Fetch portfolio data
echo "📊 Fetching portfolio data..."
PORTFOLIO=$(node scripts/fetch-portfolio.js "$WALLET" "$CHAIN")

# Parse totals
TOTAL_VALUE=$(echo "$PORTFOLIO" | jq -r '.totalValue')
echo "Total Portfolio Value: \$$TOTAL_VALUE"
echo ""

# Asset breakdown
echo "=== Asset Breakdown ==="
echo "$PORTFOLIO" | jq -r '.breakdown | to_entries[] | "\(.key): $\(.value) (\(.value / '$TOTAL_VALUE' * 100 | floor)%)"'
echo ""

# Risk analysis
echo "=== Risk Analysis ==="
RISK_SCORE=$(node scripts/calculate-risk.js "$PORTFOLIO")
echo "Overall Risk Score: $RISK_SCORE / 100"

if (( $(echo "$RISK_SCORE < 30" | bc -l) )); then
  echo "Risk Level: 🟢 LOW (Conservative)"
elif (( $(echo "$RISK_SCORE < 60" | bc -l) )); then
  echo "Risk Level: 🟡 MODERATE"
else
  echo "Risk Level: 🔴 HIGH (Degen)"
fi
echo ""

# Concentration analysis
echo "=== Top Holdings ==="
echo "$PORTFOLIO" | jq -r '.topHoldings[] | "  \(.symbol): $\(.value) (\(.percentage)%)"'
echo ""

# Recommendations
echo "=== Optimization Recommendations ==="
RECS=$(node scripts/generate-recommendations.js "$PORTFOLIO")
echo "$RECS" | jq -r '.[] | "  • \(.)"'
echo ""

# Liquidation risk
LIQUIDATION_RISK=$(echo "$PORTFOLIO" | jq -r '.liquidationRisk // 0')
if (( $(echo "$LIQUIDATION_RISK > 50" | bc -l) )); then
  echo "⚠️  WARNING: High liquidation risk detected!"
  echo "   Consider adding collateral or reducing leverage"
  echo ""
fi

# Summary
echo "=== Summary ==="
echo "Portfolio analyzed successfully ✅"
echo "Powered by \$BANKR token"
