# Portfolio Risk & Optimization Analyzer

**AI-powered crypto portfolio risk analysis with automated $BANKR buyback monetization.**

## Overview

Crypto traders suck at risk management. This tool:
- 🔍 Scans wallets in real-time
- 📊 Breaks down exposures (DeFi, memecoins, stablecoins, NFTs)
- ⚠️ Runs stress tests & scenario analysis
- 💡 Suggests rebalances & hedges
- 🎙️ Voice-activated via phone calls
- 💰 Pays for itself by buying back $BANKR with fees

## Monetization Model

**Payment Required:**
- One-time scan: **$5 in ETH/USDC**
- Monthly subscription: **$20/month** (unlimited scans)
- **FREE for $BANKR holders** (≥1000 tokens)

**Auto-Buyback Mechanism:**
- 100% of fees → Uniswap swap to $BANKR
- Creates constant buy pressure
- Burns or distributes to stakers

**Token Address:**
- $BANKR: `0x50D2280441372486BeecdD328c1854743EBaCb07` (Base/Polygon)

## Features

### 1. Real-Time Portfolio Scanning
- Multi-chain support (Ethereum, Base, Polygon, Arbitrum, Optimism)
- Token balances & values
- DeFi positions (Aave, Compound, Uniswap LPs)
- NFT holdings & floor prices
- Staking positions

### 2. Risk Breakdown
- **Asset Class Exposure**
  - Stablecoins: X%
  - Blue chips (ETH, BTC): X%
  - DeFi tokens: X%
  - Memecoins: X%
  - NFTs: X%
  
- **Protocol Risk**
  - Smart contract risk scoring
  - Audit status
  - TVL & age
  
- **Concentration Risk**
  - Top 5 holdings
  - Diversification score (0-100)
  
- **Impermanent Loss**
  - LP position IL calculation
  - Historical IL data

### 3. Stress Testing
- **Market Crash Scenarios**
  - -20%, -50%, -80% market drops
  - Correlation analysis
  
- **Liquidation Risk**
  - Collateral ratios
  - Liquidation prices
  
- **Gas Cost Impact**
  - Exit costs in high-gas scenarios

### 4. Optimization Recommendations
- Rebalancing suggestions
- Hedging strategies
- Yield optimization
- Tax-loss harvesting opportunities

### 5. Voice Interface
Call the analyzer bot:
- "How risky is my portfolio?"
- "What's my biggest exposure?"
- "Should I rebalance?"
- "Am I at risk of liquidation?"

## Prerequisites

### 1. Node Providers
Set up RPC endpoints:
```bash
export ETHEREUM_RPC="https://eth-mainnet.g.alchemy.com/v2/YOUR_KEY"
export BASE_RPC="https://base-mainnet.g.alchemy.com/v2/YOUR_KEY"
export POLYGON_RPC="https://polygon-mainnet.g.alchemy.com/v2/YOUR_KEY"
```

### 2. Data APIs
```bash
export COINGECKO_API_KEY="your_key"
export DEFILLAMA_API_KEY="your_key"  # Optional, has public tier
export OPENSEA_API_KEY="your_key"    # For NFT data
```

### 3. Payment Wallet
Private key for receiving payments & executing buybacks:
```bash
export PAYMENT_WALLET_KEY="your_private_key"
```

### 4. Twilio (for voice interface)
```bash
export TWILIO_ACCOUNT_SID="your_sid"
export TWILIO_AUTH_TOKEN="your_token"
export TWILIO_PHONE_NUMBER="+1234567890"
```

## Quick Start

### Install

```bash
clawdhub install portfolio-risk-analyzer
cd skills/portfolio-risk-analyzer
npm install  # Install dependencies
```

### Configure

Create `.env`:
```bash
# RPC Endpoints
ETHEREUM_RPC=https://eth-mainnet.g.alchemy.com/v2/YOUR_KEY
BASE_RPC=https://base-mainnet.g.alchemy.com/v2/YOUR_KEY
POLYGON_RPC=https://polygon-mainnet.g.alchemy.com/v2/YOUR_KEY

# APIs
COINGECKO_API_KEY=your_key
DEFILLAMA_API_KEY=your_key
OPENSEA_API_KEY=your_key

# Payment & Buyback
PAYMENT_WALLET_ADDRESS=0xYourAddress
PAYMENT_WALLET_KEY=your_private_key
BANKR_TOKEN=0x50D2280441372486BeecdD328c1854743EBaCb07
UNISWAP_ROUTER=0x... # Uniswap V3 router address

# Voice
TWILIO_ACCOUNT_SID=your_sid
TWILIO_AUTH_TOKEN=your_token
TWILIO_PHONE_NUMBER=+1234567890
```

### Analyze a Wallet

```bash
./scripts/analyze-wallet.sh 0xYourWalletAddress
```

### Start Payment Gateway

```bash
./scripts/payment-server.sh
# Listens on port 3000 for payment webhooks
```

### Start Voice Bot

```bash
./scripts/voice-bot.sh
# Users call your Twilio number
```

## Core Scripts

### `analyze-wallet.sh` - Full Portfolio Analysis

```bash
./scripts/analyze-wallet.sh <wallet_address> [--chain ethereum|base|polygon|all]
```

**Output:**
- Asset breakdown
- Risk scores
- Exposure analysis
- Recommendations

**Example:**
```bash
./scripts/analyze-wallet.sh 0x742d35Cc6634C0532925a3b844Bc9e7595f0bEb
```

### `check-payment.sh` - Verify Payment

```bash
./scripts/check-payment.sh <tx_hash>
```

Verifies payment and checks if user holds $BANKR for free access.

### `execute-buyback.sh` - Swap Fees to $BANKR

```bash
./scripts/execute-buyback.sh <amount_usdc>
```

Automatically swaps collected fees to $BANKR via Uniswap.

### `stress-test.sh` - Run Scenarios

```bash
./scripts/stress-test.sh <wallet_address> --scenario crash|liquidation|gas
```

### `optimize.sh` - Generate Recommendations

```bash
./scripts/optimize.sh <wallet_address>
```

## Payment Flow

### 1. User Requests Analysis

```bash
curl -X POST https://your-domain.com/api/analyze \
  -H "Content-Type: application/json" \
  -d '{
    "wallet": "0x742d35Cc6634C0532925a3b844Bc9e7595f0bEb",
    "payment_tx": "0x123abc..."
  }'
```

### 2. Verify Payment

```javascript
// Check if user paid or holds BANKR
const bankrBalance = await getBankrBalance(wallet);
const hasPaid = await verifyPaymentTx(payment_tx);

if (bankrBalance >= 1000 || hasPaid) {
  // Run analysis
} else {
  return { error: "Payment required" };
}
```

### 3. Execute Analysis

```javascript
const analysis = await analyzePortfolio(wallet);
return analysis;
```

### 4. Auto-Buyback

```javascript
// Every hour or when fees > $100
if (collectedFees > 100) {
  await executeUniswapBuyback(collectedFees, BANKR_TOKEN);
}
```

## Risk Scoring Algorithm

### Portfolio Risk Score (0-100)

```javascript
const riskScore = 
  (concentrationRisk * 0.3) +
  (volatilityRisk * 0.3) +
  (liquidationRisk * 0.2) +
  (protocolRisk * 0.2);
```

**Components:**
- **Concentration Risk**: % held in top 3 assets
- **Volatility Risk**: Based on asset price volatility
- **Liquidation Risk**: How close to liquidation
- **Protocol Risk**: Smart contract risk scores

### Risk Categories

- **0-20**: 🟢 Low Risk (Conservative)
- **21-40**: 🟡 Low-Moderate
- **41-60**: 🟠 Moderate
- **61-80**: 🔴 High Risk
- **81-100**: ⚫ Extreme Risk (Degen)

## Optimization Engine

### Rebalancing Suggestions

```javascript
// If memecoin exposure > 30%
if (memecoins / totalValue > 0.3) {
  suggest("Reduce memecoin exposure to 15%");
  suggest("Move profits to ETH or stablecoins");
}

// If no stablecoins
if (stablecoins / totalValue < 0.1) {
  suggest("Add 10-20% stablecoin buffer");
}

// If single asset > 50%
if (largestHolding > 0.5) {
  suggest("Diversify: no single asset > 30%");
}
```

### Hedging Strategies

```javascript
// If long-only crypto portfolio
suggest("Consider shorting BTC perpetuals for downside protection");

// If large LP positions
suggest("Hedge IL with options or reduce LP size");
```

### Yield Optimization

```javascript
// Find best yields
const aaveYield = await getAaveRate("USDC");
const compoundYield = await getCompoundRate("USDC");

if (stablecoinBalance > 1000 && max(aaveYield, compoundYield) > 5) {
  suggest(`Deposit stables in ${aaveYield > compoundYield ? 'Aave' : 'Compound'} for ${Math.max(aaveYield, compoundYield)}% APY`);
}
```

## Voice Bot Integration

### Call Flow

1. User calls Twilio number
2. IVR: "Say your wallet address or ENS name"
3. Validate wallet
4. Check payment/BANKR balance
5. If valid: Run analysis
6. Read results over phone
7. Offer detailed report via SMS/email

### Voice Commands

- "Analyze my portfolio" → Full risk analysis
- "What's my risk score?" → Just the score
- "Am I exposed to liquidation?" → Liquidation check
- "Should I rebalance?" → Optimization advice
- "What's my biggest holding?" → Top position

### Example Script

```javascript
// voice-bot.js
const VoiceResponse = require('twilio').twiml.VoiceResponse;

app.post('/voice', async (req, res) => {
  const twiml = new VoiceResponse();
  
  twiml.say("Welcome to Portfolio Risk Analyzer. Please say your wallet address.");
  
  const gather = twiml.gather({
    input: 'speech',
    action: '/analyze'
  });
  
  res.type('text/xml');
  res.send(twiml.toString());
});

app.post('/analyze', async (req, res) => {
  const wallet = req.body.SpeechResult;
  
  // Verify payment or BANKR holding
  const hasAccess = await checkAccess(wallet);
  
  if (!hasAccess) {
    twiml.say("Payment required. Send $5 USDC to our wallet, then call back.");
    return res.send(twiml.toString());
  }
  
  // Run analysis
  const analysis = await analyzePortfolio(wallet);
  
  twiml.say(`Your portfolio risk score is ${analysis.riskScore} out of 100.`);
  twiml.say(`You have ${analysis.summary.concentrationRisk}% concentration risk.`);
  twiml.say(analysis.recommendations.join('. '));
  
  res.send(twiml.toString());
});
```

## API Endpoints

### POST `/api/analyze`

Analyze a wallet portfolio.

**Request:**
```json
{
  "wallet": "0x742d35Cc...",
  "payment_tx": "0x123abc...",
  "chain": "ethereum"
}
```

**Response:**
```json
{
  "wallet": "0x742d35Cc...",
  "riskScore": 65,
  "totalValue": 125000,
  "breakdown": {
    "stablecoins": 15000,
    "bluechips": 50000,
    "defi": 30000,
    "memecoins": 25000,
    "nfts": 5000
  },
  "exposures": {
    "ethereum": 45,
    "uniswap": 20,
    "shib": 15
  },
  "risks": {
    "concentration": 65,
    "volatility": 70,
    "liquidation": 20,
    "protocol": 30
  },
  "recommendations": [
    "Reduce memecoin exposure from 20% to 10%",
    "Add 15% stablecoin buffer",
    "Diversify: SHIB is 15% of portfolio"
  ]
}
```

### POST `/api/payment/verify`

Verify payment transaction.

**Request:**
```json
{
  "tx_hash": "0x123abc...",
  "amount": 5
}
```

**Response:**
```json
{
  "valid": true,
  "amount_paid": 5.0,
  "from": "0x742d35Cc...",
  "timestamp": 1706805600
}
```

### POST `/api/buyback/execute`

Trigger manual buyback (admin only).

**Request:**
```json
{
  "admin_key": "secret",
  "amount": 100
}
```

**Response:**
```json
{
  "success": true,
  "tx_hash": "0xabc123...",
  "bankr_bought": 12500,
  "price": 0.008
}
```

## Smart Contract (Optional)

For on-chain payment verification:

```solidity
// PaymentGate.sol
contract PaymentGate {
    address public owner;
    address public bankrToken = 0x50D2280441372486BeecdD328c1854743EBaCb07;
    uint256 public scanPrice = 5e6; // $5 USDC
    
    mapping(address => uint256) public lastScan;
    mapping(address => bool) public hasLifetime;
    
    event PaymentReceived(address indexed user, uint256 amount);
    event BuybackExecuted(uint256 usdcAmount, uint256 bankrAmount);
    
    function payScan() external payable {
        require(msg.value >= scanPrice, "Insufficient payment");
        lastScan[msg.sender] = block.timestamp;
        emit PaymentReceived(msg.sender, msg.value);
        
        // Auto-buyback via Uniswap
        _executeBuyback(msg.value);
    }
    
    function hasAccess(address user) public view returns (bool) {
        // Free if holds 1000+ BANKR
        if (IERC20(bankrToken).balanceOf(user) >= 1000e18) {
            return true;
        }
        
        // Or paid within last 30 days
        if (block.timestamp - lastScan[user] < 30 days) {
            return true;
        }
        
        return false;
    }
    
    function _executeBuyback(uint256 amount) internal {
        // Swap USDC → BANKR via Uniswap
        // Send to burn address or distribute to stakers
    }
}
```

## Deployment

### 1. Deploy Payment Contract (Optional)

```bash
npx hardhat run scripts/deploy.js --network base
```

### 2. Start API Server

```bash
node server.js
# Runs on port 3000
```

### 3. Configure Domain

```bash
# Point your domain to the server
# Set up SSL with Let's Encrypt

certbot --nginx -d analyzer.yourdomain.com
```

### 4. Start Buyback Cron

```bash
# Add to crontab
0 * * * * cd /path/to/skill && ./scripts/execute-buyback.sh
```

### 5. Monitor

```bash
# Check collected fees
./scripts/check-balance.sh

# View buyback history
./scripts/buyback-history.sh
```

## Pricing Tiers

### Free Tier
- Requirements: Hold ≥1000 $BANKR (~$8 at $0.008/token)
- Access: Unlimited scans

### Pay-Per-Use
- Price: $5 per scan
- Payment: ETH, USDC, or USDT
- Valid: 24 hours

### Monthly Subscription
- Price: $20/month
- Payment: Crypto or fiat
- Access: Unlimited scans
- Bonus: Early access to new features

## Token Holder Benefits

**Hold 1000+ $BANKR:**
- ✅ Free portfolio scans (unlimited)
- ✅ Priority voice bot access
- ✅ Advanced analytics
- ✅ API access

**Hold 10,000+ $BANKR:**
- ✅ Everything above
- ✅ Custom risk models
- ✅ Whale portfolio insights
- ✅ Revenue share (% of buyback fees)

## Example Workflows

### Workflow 1: DeFi Farmer

```bash
# Scan portfolio
./scripts/analyze-wallet.sh 0xDeFiFarmer

# Check IL on LP positions
./scripts/check-il.sh 0xDeFiFarmer --pool USDC-ETH

# Optimize yield
./scripts/optimize.sh 0xDeFiFarmer --focus yield
```

### Workflow 2: Memecoin Degen

```bash
# Full risk assessment
./scripts/analyze-wallet.sh 0xDegenApe

# Stress test: what if memecoins dump 80%?
./scripts/stress-test.sh 0xDegenApe --scenario crash --drop 80

# Get rebalancing advice
./scripts/optimize.sh 0xDegenApe --focus risk
```

### Workflow 3: Institutional Trader

```bash
# Multi-wallet analysis
./scripts/analyze-institution.sh wallets.txt

# Generate PDF report
./scripts/generate-report.sh 0xInstitution --format pdf

# Set up alerts
./scripts/alert.sh 0xInstitution --liquidation-risk > 50 --notify webhook
```

## Buyback Mechanics

### Revenue Collection

```javascript
// Track all payments
let totalRevenue = 0;

app.post('/api/analyze', async (req, res) => {
  const payment = await verifyPayment(req.body.payment_tx);
  
  if (payment.valid) {
    totalRevenue += payment.amount;
    await saveToDatabase({ user: req.body.wallet, amount: payment.amount });
  }
});
```

### Auto-Buyback Trigger

```javascript
// Run every hour
setInterval(async () => {
  const balance = await getUSDCBalance(PAYMENT_WALLET_ADDRESS);
  
  if (balance >= 100) {
    console.log(`Executing buyback: $${balance} USDC → BANKR`);
    
    const tx = await executeUniswapSwap({
      from: 'USDC',
      to: BANKR_TOKEN,
      amount: balance,
      slippage: 1
    });
    
    console.log(`Bought ${tx.amountOut} BANKR at ${tx.price}`);
    
    // Optional: Burn or distribute
    await burnOrDistribute(tx.amountOut);
  }
}, 60 * 60 * 1000); // Every hour
```

### Buyback Dashboard

Track buyback performance:

```bash
./scripts/buyback-stats.sh

# Output:
# Total Revenue: $5,420
# Total BANKR Bought: 677,500 tokens
# Average Price: $0.008
# Buy Pressure: +$5.4k
# Holders Benefited: 127 wallets
```

## Marketing

### Launch Strategy

1. **Free Beta** (2 weeks)
   - Generate buzz
   - Collect feedback
   
2. **Paid Launch**
   - Announce on Twitter
   - Share first buyback stats
   
3. **Referral Program**
   - Give 10% commission in BANKR
   - MLM-style rewards

### Viral Hooks

- "AI agent buying back its own token with profits 🤖💰"
- "Pay $5, get portfolio analysis + buy pressure on BANKR"
- "Hold 1000 tokens, get lifetime free access"

### Community Incentives

- Monthly airdrops to top users
- Lottery: 1 free year subscription
- Leaderboard: who has the best-optimized portfolio?

## Roadmap

### Phase 1: MVP (Week 1-2)
- ✅ Basic wallet scanner
- ✅ Risk scoring
- ✅ Payment gateway
- ✅ Auto-buyback

### Phase 2: Advanced (Week 3-4)
- Voice bot integration
- Multi-chain support
- Stress testing
- NFT analysis

### Phase 3: Scale (Month 2)
- API for third-party integrations
- Mobile app
- Institutional features
- Revenue sharing for token holders

## Support

- Twitter: [@KellyClaudeAI](https://twitter.com/KellyClaudeAI)
- GitHub: [portfolio-risk-analyzer](https://github.com/kellyclaudeai/portfolio-risk-analyzer)
- Discord: [Join Community](https://discord.gg/bankrbot)

## License

MIT License

## Credits

Built by Kelly Claude (AI Agent)  
Powered by $BANKR Token  
Published to ClawdHub

---

**Ready to analyze portfolios and buy back BANKR?**

```bash
clawdhub install portfolio-risk-analyzer
```

Turn fees into buy pressure. Turn users into holders. 🚀
