"""Explain recent PaperBot fills with Binance context (best-effort).

Reads: workspace/polymarket_paperbot/state/events.jsonl
Prints last N fills with: ts, token, side, px, reason, fair_up, z, against_trend.

Note: Binance rate limits apply; keep N small (e.g. 20-50).
"""

import argparse
import json
import math
from datetime import datetime, timezone
from dateutil import parser as dateparser
import urllib.request
from urllib.parse import urlencode

BINANCE = "https://api.binance.com"


def norm_cdf(x: float) -> float:
    return 0.5 * (1.0 + math.erf(x / (2.0**0.5)))


def fetch_json(url: str) -> object:
    with urllib.request.urlopen(url, timeout=20) as r:
        return json.loads(r.read().decode("utf-8"))


def binance_1m_closes(limit: int = 60) -> list[float]:
    url = BINANCE + "/api/v3/klines?" + urlencode({"symbol": "BTCUSDT", "interval": "1m", "limit": int(limit)})
    k = fetch_json(url)
    return [float(row[4]) for row in k]


def realized_sigma_1m(closes: list[float]) -> float | None:
    if len(closes) < 3:
        return None
    rets = []
    for a, b in zip(closes[:-1], closes[1:]):
        if a <= 0:
            continue
        rets.append((b - a) / a)
    if len(rets) < 3:
        return None
    mu = sum(rets) / len(rets)
    var = sum((x - mu) ** 2 for x in rets) / (len(rets) - 1)
    return var ** 0.5


def binance_spot() -> float | None:
    url = BINANCE + "/api/v3/ticker/price?" + urlencode({"symbol": "BTCUSDT"})
    j = fetch_json(url)
    try:
        return float(j.get("price"))
    except Exception:
        return None


def binance_1h_open_at(start_ms: int) -> float | None:
    url = BINANCE + "/api/v3/klines?" + urlencode({"symbol": "BTCUSDT", "interval": "1h", "startTime": int(start_ms), "limit": 1})
    k = fetch_json(url)
    if not k:
        return None
    return float(k[0][1])


def estimate_p_up(open_px: float, spot: float, sigma_1m: float | None, minutes_left: float) -> tuple[float, float]:
    if open_px <= 0:
        return 0.5, 0.0
    cur_ret = (spot - open_px) / open_px
    if sigma_1m is None or sigma_1m <= 0 or minutes_left <= 0:
        return (1.0 if cur_ret > 0 else (0.0 if cur_ret < 0 else 0.5)), 0.0
    stdev = sigma_1m * math.sqrt(max(0.1, minutes_left))
    z = cur_ret / max(1e-9, stdev)
    return max(0.0, min(1.0, norm_cdf(z))), z


def main() -> int:
    ap = argparse.ArgumentParser()
    ap.add_argument("--events", default=r"C:\\Users\\domin\\.openclaw\\workspace\\polymarket_paperbot\\state\\events.jsonl")
    ap.add_argument("--n", type=int, default=30)
    ap.add_argument("--sec_to_end", type=float, default=1800.0, help="rough minutes_left estimate if you don't know exact")
    args = ap.parse_args()

    fills = []
    with open(args.events, "r", encoding="utf-8") as f:
        for line in f:
            try:
                j = json.loads(line)
            except Exception:
                continue
            if j.get("type") == "fill":
                fills.append(j)

    fills = fills[-int(args.n):]

    closes = binance_1m_closes(60)
    sigma = realized_sigma_1m(closes)
    spot = binance_spot()

    print("ts\tside\ttoken\tpx\treason\tfair_up\tz\tagainst_trend")

    for j in fills:
        ts = j.get("ts")
        token = j.get("token")
        side = j.get("side")
        px = float(j.get("px") or 0.0)
        reason = j.get("reason")

        fair_up = None
        z = 0.0
        against = "?"

        if spot is not None and ts:
            dt = dateparser.parse(ts).astimezone(timezone.utc)
            hour = dt.replace(minute=0, second=0, microsecond=0)
            open_px = binance_1h_open_at(int(hour.timestamp() * 1000))
            if open_px is None:
                open_px = closes[0] if closes else spot
            minutes_left = max(0.1, float(args.sec_to_end) / 60.0)
            fair_up, z = estimate_p_up(open_px, spot, sigma, minutes_left)
            if "Up" in str(token):
                against = "YES" if z < -0.25 else "NO"
            if "Down" in str(token):
                against = "YES" if z > 0.25 else "NO"

        fu = "" if fair_up is None else f"{fair_up:.3f}"
        print(f"{ts}\t{side}\t{token}\t{px:.4f}\t{reason}\t{fu}\t{z:.2f}\t{against}")

    return 0


if __name__ == "__main__":
    raise SystemExit(main())
