---
name: polymarket-setup
description: Set up automated trading on Polymarket. Covers wallet setup, token approvals, API authentication, market discovery, order placement, WebSocket feeds, and position management.
metadata:
  {
    "openclaw": {
      "requires": {
        "env": [
          "POLYMARKET_PRIVATE_KEY",
          "POLYMARKET_PUBLIC_ADDRESS",
          "POLYMARKET_PROXY_ADDRESS",
          "POLYMARKET_SIGNATURE_TYPE",
          "POLYMARKET_API_KEY",
          "POLYMARKET_API_SECRET",
          "POLYMARKET_API_PASSPHRASE"
        ],
        "optionalEnv": [
          "POLYMARKET_BUILDER_API_KEY",
          "POLYMARKET_BUILDER_SECRET",
          "POLYMARKET_BUILDER_PASSPHRASE"
        ]
      }
    }
  }
---

# Skill: Polymarket Trading Setup

Use this skill when the user wants to:
- Set up automated trading on Polymarket
- Build a Polymarket trading bot from scratch
- Configure wallet, API credentials, or token approvals for Polymarket
- Understand how to connect to Polymarket's APIs
- Debug issues with an existing Polymarket bot setup

The full technical reference is in `GUIDE.md` (same directory as this file). Read it before starting.

---

## What You're Setting Up

Polymarket trading requires four things in order:

1. **A funded proxy wallet** with USDC on Polygon
2. **Token approvals** (USDC spend approved for exchange contracts)
3. **CLOB API credentials** (derived from the wallet, stored in env)
4. **Connectivity verified** across all four API surfaces

---

## Step-by-Step Agent Instructions

### Step 1: Check What Already Exists

Before doing anything, check:
- Is there an `.env` file with `POLYMARKET_PRIVATE_KEY` and `POLYMARKET_PROXY_ADDRESS`?
- Is there a `config.json` with Polymarket settings?
- Is there an existing bot directory to work within?

If credentials already exist, load and validate them rather than starting from scratch.

### Step 2: Environment Setup

If starting fresh, create a `.env` file with:

```bash
POLYMARKET_PRIVATE_KEY=0x...
POLYMARKET_PUBLIC_ADDRESS=0x...     # proxy wallet address
POLYMARKET_PROXY_ADDRESS=0x...      # same as PUBLIC_ADDRESS for type 2
POLYMARKET_SIGNATURE_TYPE=2
POLYMARKET_WEBSOCKET_URL=wss://ws-subscriptions-clob.polymarket.com
POLYMARKET_DATA_API=https://data-api.polymarket.com
```

The proxy wallet address comes from the user's Polymarket account settings page.

### Step 3: Install Dependencies

```bash
pip install "py-clob-client>=0.28.0" httpx "websocket-client>=1.9.0" orjson pandas python-dotenv
```

Or add to `pyproject.toml` and run `uv sync`.

### Step 4: Token Approvals

**Via UI (recommended for new users):** Deposit USDC through the Polymarket web app — approvals happen automatically.

**Headless (server deployment):** Use the programmatic approval flow from GUIDE.md Section 4. This requires Polymarket Builder API credentials (separate from CLOB creds).

The four contracts that need approval are listed in GUIDE.md Section 2.

### Step 5: Derive and Persist API Credentials

```python
from py_clob_client.client import ClobClient

client = ClobClient(
    "https://clob.polymarket.com",
    key=os.getenv("POLYMARKET_PRIVATE_KEY"),
    chain_id=137,
    signature_type=int(os.getenv("POLYMARKET_SIGNATURE_TYPE", "2")),
    funder=os.getenv("POLYMARKET_PROXY_ADDRESS"),
)
creds = client.derive_api_key()
# Write credentials to .env — do not log or print them
```

Add to `.env`:

```bash
POLYMARKET_API_KEY=...
POLYMARKET_API_SECRET=...
POLYMARKET_API_PASSPHRASE=...
```

On subsequent startups, load from env instead of re-deriving (see GUIDE.md Section 3).

### Step 6: Verify Connectivity

Test each surface in order. Stop and diagnose if any step fails.

```python
import httpx, json

# 1. Gamma API
event = httpx.get("https://gamma-api.polymarket.com/events/slug/bitcoin-price-on-february-11").json()
print(f"Gamma OK: {event.get('title')}")

# 2. CLOB REST - order book
book = httpx.get("https://clob.polymarket.com/book", params={"token_id": "<any_token_id>"}).json()
print(f"CLOB OK: {len(book.get('bids', []))} bids, {len(book.get('asks', []))} asks")

# 3. Data API - positions
positions = httpx.get(
    "https://data-api.polymarket.com/positions",
    params={"user": os.getenv("POLYMARKET_PROXY_ADDRESS")}
).json()
print(f"Data API OK: {len(positions)} open positions")
```

### Step 7: Place a Test Order

Place a 2-share order at a far-from-market price (very low probability on a real market) to verify the full signing and posting flow without risk of a fill:

```python
from py_clob_client.clob_types import OrderArgs, OrderType
from py_clob_client.order_builder.constants import BUY

# Use a real token ID from Step 6, price far from market
order = OrderArgs(price=0.02, side=BUY, size=2, token_id="<token_id>")
signed = client.create_order(order)
resp = client.post_order(signed, OrderType.FAK)
print(resp)  # Should show success: true
```

---

## Key Facts to Remember

- **Chain:** Polygon mainnet (ID: 137)
- **Currency:** USDC (`0x3c499c542cEF5E3811e1192ce70d8cC03d5c3359`)
- **Signature type 2** is standard for programmatic trading
- **Proxy address ≠ signing key address** — positions live on proxy, orders signed by EOA
- **clobTokenIds[0] = YES, clobTokenIds[1] = NO** — never mix these up
- **Minimum order value: $1.00** — `size * price >= 1.0`
- **Order size must be integer** — always `int()`
- **Update positions only on MINED, not MATCHED** — MATCHED can fail

---

## Common First-Time Failures

| Symptom | Likely Cause |
|---|---|
| `insufficient balance` | Wrong address used (EOA instead of proxy), or USDC not deposited |
| Order silently rejected | Price has too many decimal places — round to 2dp (or 3dp if price < 0.04 or > 0.96) |
| Order value error | `size * price < 1.0` |
| WebSocket never sends data | Wrong subscription message format, or using array instead of object |
| Positions always empty | Querying EOA address instead of proxy wallet address |
| Stale positions after trade | Updating on MATCHED instead of MINED |

---

## Reference

Full API reference, all code patterns, and detailed explanations: **GUIDE.md** (same directory).
