---
name: polymarket-auto-trader
description: Autonomous Polymarket prediction market trading agent. Scans markets, evaluates probabilities with LLM, sizes positions with Kelly criterion, and executes trades via CLOB API. Use when user wants to trade on Polymarket, set up automated prediction market trading, or build a trading bot. Supports cron-based autonomous operation, P&L tracking, and budget management.
metadata: {"openclaw": {"requires": {"env": ["PRIVATE_KEY", "LLM_API_KEY"]}, "primaryEnv": "LLM_API_KEY", "homepage": "https://github.com/srikanthbellary"}}
---

# Polymarket Auto-Trader

Fully autonomous prediction market trading agent for Polymarket. Evaluates markets using LLM probability estimation, sizes positions with Kelly criterion, and executes trades via the Polymarket CLOB API from a non-US VPS.

## Prerequisites

- **Non-US VPS** — Polymarket blocks US IPs. Use DigitalOcean Amsterdam, Hetzner EU, etc.
- **Polygon wallet** with USDC.e (bridged USDC, NOT native USDC)
- **MATIC** for gas (~0.1 MATIC sufficient for hundreds of trades)
- **Anthropic API key** (uses Haiku at ~$0.001/evaluation)

## Setup

### 1. VPS Environment

SSH into your non-US VPS and run:

```bash
python3 {baseDir}/scripts/setup_vps.sh
```

Or manually:
```bash
apt update && apt install -y python3 python3-venv
python3 -m venv /opt/trader
/opt/trader/bin/pip install py-clob-client python-dotenv web3 requests
```

### 2. Configuration

Create `/opt/trader/app/.env`:
```
PRIVATE_KEY=<your-polygon-wallet-private-key>
LLM_API_KEY=<your-anthropic-api-key>
```

### 3. Blockchain Approvals

Before trading, approve USDC.e and CTF tokens for Polymarket contracts. Run:
```bash
python3 {baseDir}/scripts/approve_contracts.py
```

Required approvals (6 total):
- USDC.e → CTF Exchange, Neg Risk Exchange, Neg Risk Adapter
- CTF → CTF Exchange, Neg Risk Exchange, Neg Risk Adapter

### 4. Deploy Trading Script

```bash
cp {baseDir}/scripts/run_trade.py /opt/trader/app/
cp {baseDir}/scripts/pnl_tracker.py /opt/trader/app/
```

### 5. Cron Automation

```bash
crontab -e
# Add: */10 * * * * cd /opt/trader/app && /opt/trader/bin/python3 run_trade.py >> cron.log 2>&1
```

## How It Works

1. **Market Scan** — Fetches active markets from Gamma API, filters by liquidity and time horizon
2. **LLM Evaluation** — Asks Claude Haiku to estimate true probability for each market
3. **Edge Detection** — Compares LLM fair value vs market price (min 5% edge threshold)
4. **Kelly Sizing** — Half-Kelly criterion with 25% max position size cap
5. **Order Execution** — Places limit orders via CLOB API with GTC (good-till-cancelled)
6. **Dedup** — Tracks all trades in `trades.jsonl`, skips already-traded markets
7. **Budget Control** — Tracks LLM inference costs separately from trading capital

## Trading Parameters

Configurable in `run_trade.py`:
- `EDGE_THRESHOLD` — Minimum edge to trade (default: 0.05 = 5%)
- `MIN_SHARES` — Minimum order size (Polymarket requires ≥5 shares)
- Bankroll allocation: 80% usable, 25% max per trade, 30% cap per single position
- Market horizon: Prioritizes markets ending within 30 days

## Monitoring

Check P&L anytime:
```bash
python3 /opt/trader/app/pnl_tracker.py
```

Check recent activity:
```bash
tail -50 /opt/trader/app/cron.log
```

## Key Technical Details

- **Wallet type:** EOA (signature_type=0), NOT proxy wallet
- **Token:** USDC.e (`0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174`), not native USDC
- **Neg-risk markets** (elections, sports leagues) require USDC.e approval for Neg Risk Adapter (`0xd91E80cF2E7be2e162c6513ceD06f1dD0dA35296`)
- **US geoblock** — All API calls must originate from non-US IP. VPN insufficient; use actual non-US VPS.

## Cost

- LLM inference: ~$0.001 per market evaluation (Haiku)
- Typical cycle (40 evals): ~$0.04
- Gas: negligible on Polygon (~$0.001 per trade)

## ⚠️ Security Considerations

- **Use a DEDICATED wallet with minimal funds.** Never use your main wallet's private key. Create a fresh wallet and fund it only with what you're willing to risk.
- **PRIVATE_KEY is stored on disk** in `.env`. Harden your VPS: strict file permissions (`chmod 600 .env`), no shared access, firewall, SSH keys only.
- **MAX_UINT approvals** are standard in DeFi but grant broad spending rights. The approved contracts are official Polymarket contracts. Review addresses in `references/contract-addresses.md` before running.
- **Test with tiny amounts first** ($5-10) before scaling up.
- **Monitor actively** — check `cron.log` and run `pnl_tracker.py` regularly.
- **LLM_API_KEY billing** — each cycle costs ~$0.04 (Haiku). Set billing alerts on your Anthropic account.
- **This is autonomous trading software.** Bugs or market conditions can cause losses. Use at your own risk.

## References

- See `references/polymarket-api.md` for full CLOB API documentation
- See `references/contract-addresses.md` for all Polygon contract addresses
