"""
x402 Payment Verification for Base L2
Handles payment verification for API access
"""

import json
import os
import time
from urllib.request import urlopen, Request
from urllib.error import URLError
from typing import Optional, Dict, Any, Tuple

# Configuration
PAYMENT_WALLET = os.environ.get("PAYMENT_WALLET", "0xB8B984d8150571D2Dd19aF2400D52332E262c945")
PRICE_USDC = float(os.environ.get("PRICE_USDC", "0.05"))  # $0.05 per request
USDC_CONTRACT = "0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913"  # USDC on Base
# Multiple RPC endpoints for fallback
BASE_RPCS = [
    os.environ.get("BASE_RPC", "https://base.llamarpc.com"),
    "https://base.drpc.org",
    "https://1rpc.io/base",
    "https://base-mainnet.public.blastapi.io",
]

# Cache for verified payments (tx_hash -> timestamp)
# In production, use Redis or similar
VERIFIED_PAYMENTS: Dict[str, float] = {}
PAYMENT_TTL = 60  # 1 minute - payment valid for multiple requests


def get_payment_instructions() -> Dict[str, Any]:
    """Return payment instructions for 402 response."""
    return {
        "x402": {
            "version": "1.0",
            "network": "base",
            "accepts": [{
                "scheme": "exact",
                "network": "base",
                "maxAmountRequired": str(int(PRICE_USDC * 1e6)),  # USDC has 6 decimals
                "resource": PAYMENT_WALLET,
                "description": f"Pay {PRICE_USDC} USDC to access PolyEdge API",
                "mimeType": "application/json",
                "payTo": PAYMENT_WALLET,
                "maxTimeoutSeconds": 300,
                "asset": USDC_CONTRACT,
            }],
            "error": "Payment required"
        }
    }


def verify_payment(tx_hash: str) -> Tuple[bool, str]:
    """
    Verify a payment transaction on Base.
    Returns (success, message).
    """
    if not tx_hash:
        return False, "No transaction hash provided"
    
    # Clean the hash
    tx_hash = tx_hash.strip().lower()
    if not tx_hash.startswith("0x"):
        tx_hash = "0x" + tx_hash
    
    # Check cache first
    if tx_hash in VERIFIED_PAYMENTS:
        cached_time = VERIFIED_PAYMENTS[tx_hash]
        if time.time() - cached_time < PAYMENT_TTL:
            return True, "Payment verified (cached)"
        else:
            # Expired, remove from cache
            del VERIFIED_PAYMENTS[tx_hash]
    
    # Fetch transaction receipt from Base RPC
    try:
        receipt = fetch_tx_receipt(tx_hash)
        if not receipt:
            return False, "Transaction not found"
        
        # Check if successful
        if receipt.get("status") != "0x1":
            return False, "Transaction failed"
        
        # Parse logs for USDC transfer
        transfer_valid = verify_usdc_transfer(receipt.get("logs", []))
        if not transfer_valid:
            return False, "No valid USDC transfer to payment address found"
        
        # Cache successful verification
        VERIFIED_PAYMENTS[tx_hash] = time.time()
        return True, "Payment verified"
        
    except Exception as e:
        return False, f"Verification error: {str(e)}"


def fetch_tx_receipt(tx_hash: str) -> Optional[Dict[str, Any]]:
    """Fetch transaction receipt from Base RPC with fallback."""
    payload = {
        "jsonrpc": "2.0",
        "method": "eth_getTransactionReceipt",
        "params": [tx_hash],
        "id": 1
    }
    
    for rpc_url in BASE_RPCS:
        try:
            req = Request(
                rpc_url,
                data=json.dumps(payload).encode(),
                headers={"Content-Type": "application/json"}
            )
            with urlopen(req, timeout=10) as resp:
                result = json.loads(resp.read().decode())
                if result.get("result"):
                    return result.get("result")
        except (URLError, json.JSONDecodeError) as e:
            print(f"RPC Error ({rpc_url}): {e}")
            continue
    
    return None


def verify_usdc_transfer(logs: list) -> bool:
    """
    Verify logs contain USDC transfer to our wallet.
    Transfer event: Transfer(address from, address to, uint256 value)
    Topic0: 0xddf252ad... (Transfer signature)
    """
    TRANSFER_TOPIC = "0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef"
    
    for log in logs:
        # Check if it's the USDC contract
        if log.get("address", "").lower() != USDC_CONTRACT.lower():
            continue
        
        topics = log.get("topics", [])
        if len(topics) < 3:
            continue
        
        # Check Transfer event
        if topics[0].lower() != TRANSFER_TOPIC.lower():
            continue
        
        # Topic[2] is the 'to' address (padded to 32 bytes)
        to_address = "0x" + topics[2][-40:]
        if to_address.lower() != PAYMENT_WALLET.lower():
            continue
        
        # Check amount (in log data)
        data = log.get("data", "0x0")
        amount = int(data, 16) if data else 0
        min_amount = int(PRICE_USDC * 1e6 * 0.99)  # Allow 1% slippage
        
        if amount >= min_amount:
            return True
    
    return False


def check_payment_header(headers: Dict[str, str]) -> Tuple[bool, str, Optional[str]]:
    """
    Check for payment in request headers.
    Returns (has_payment, message, tx_hash).
    
    Supported headers:
    - X-Payment: <tx_hash>
    - X-402-Payment: <tx_hash>
    - Authorization: x402 <tx_hash>
    """
    # Check various header formats
    tx_hash = None
    
    if "X-Payment" in headers:
        tx_hash = headers["X-Payment"]
    elif "X-402-Payment" in headers:
        tx_hash = headers["X-402-Payment"]
    elif "Authorization" in headers:
        auth = headers["Authorization"]
        if auth.lower().startswith("x402 "):
            tx_hash = auth[5:].strip()
    
    if not tx_hash:
        return False, "No payment provided", None
    
    verified, msg = verify_payment(tx_hash)
    return verified, msg, tx_hash
