"""
Polymarket API Client
Fetches market data from Polymarket's Gamma API
"""

import json
import re
from urllib.request import urlopen, Request
from urllib.error import URLError
from typing import Optional, Dict, Any

GAMMA_API = "https://gamma-api.polymarket.com"
USER_AGENT = "PolymarketCorrelation/1.0"


def fetch_api(endpoint: str, timeout: int = 30) -> Optional[Dict[str, Any]]:
    """Fetch data from Gamma API."""
    url = f"{GAMMA_API}{endpoint}"
    try:
        req = Request(url, headers={"User-Agent": USER_AGENT})
        with urlopen(req, timeout=timeout) as resp:
            return json.loads(resp.read().decode())
    except (URLError, json.JSONDecodeError) as e:
        print(f"API Error: {e}")
        return None


def extract_slug_from_url(url: str) -> Optional[str]:
    """Extract market slug from Polymarket URL."""
    # Pattern: polymarket.com/event/{event-slug}/{market-slug}
    match = re.search(r'polymarket\.com/event/[^/]+/([^/\?\#]+)', url)
    if match:
        return match.group(1)
    # Pattern: polymarket.com/event/{event-slug}
    match = re.search(r'polymarket\.com/event/([^/\?\#]+)', url)
    if match:
        return match.group(1)
    return None


def get_market_by_slug(slug: str) -> Optional[Dict[str, Any]]:
    """Fetch market by slug."""
    data = fetch_api(f"/markets?slug={slug}")
    if data and isinstance(data, list) and len(data) > 0:
        return data[0]
    return None


def get_market_by_id(market_id: str) -> Optional[Dict[str, Any]]:
    """Fetch market by ID."""
    return fetch_api(f"/markets/{market_id}")


def resolve_market(identifier: str) -> Optional[Dict[str, Any]]:
    """
    Resolve a market from URL, slug, or ID.
    Returns normalized market data.
    """
    # If it's a URL, extract slug
    if identifier.startswith("http"):
        slug = extract_slug_from_url(identifier)
        if slug:
            market = get_market_by_slug(slug)
            if market:
                return normalize_market(market)
    
    # Try as numeric ID
    if identifier.isdigit():
        market = get_market_by_id(identifier)
        if market:
            return normalize_market(market)
    
    # Try as slug directly
    market = get_market_by_slug(identifier)
    if market:
        return normalize_market(market)
    
    return None


def normalize_market(raw: Dict[str, Any]) -> Dict[str, Any]:
    """Normalize market data to consistent format."""
    prices = raw.get("outcomePrices", [])
    if isinstance(prices, str):
        try:
            prices = json.loads(prices)
        except json.JSONDecodeError:
            prices = []
    
    yes_price = float(prices[0]) if len(prices) > 0 else 0.0
    no_price = float(prices[1]) if len(prices) > 1 else 0.0
    
    return {
        "id": raw.get("id") or raw.get("conditionId"),
        "slug": raw.get("slug", ""),
        "question": raw.get("question", "Unknown"),
        "yes_price": yes_price,
        "no_price": no_price,
        "volume": float(raw.get("volume", 0) or 0),
        "liquidity": float(raw.get("liquidity", 0) or 0),
        "end_date": raw.get("endDate"),
        "category": classify_market(raw.get("question", "")),
        "active": raw.get("active", True),
        "closed": raw.get("closed", False),
    }


def classify_market(question: str) -> str:
    """Classify market into category for correlation lookup."""
    q = question.lower()
    
    if any(w in q for w in ["trump", "biden", "election", "president", "congress", "senate", "governor", "vote"]):
        return "politics_us"
    if any(w in q for w in ["fed", "rate", "inflation", "gdp", "recession", "unemployment"]):
        return "economics"
    if any(w in q for w in ["s&p", "nasdaq", "dow", "stock", "bitcoin", "crypto", "eth"]):
        return "markets"
    if any(w in q for w in ["war", "ukraine", "russia", "china", "taiwan", "military"]):
        return "geopolitics"
    if any(w in q for w in ["ai", "openai", "google", "meta", "apple", "microsoft"]):
        return "tech"
    if any(w in q for w in ["oscar", "grammy", "super bowl", "world cup", "nba", "nfl"]):
        return "entertainment"
    
    return "other"


def search_markets(query: str, limit: int = 10) -> list:
    """Search for markets matching a query."""
    data = fetch_api(f"/markets?active=true&closed=false&limit={limit}")
    if not data:
        return []
    
    query_lower = query.lower()
    matches = []
    for market in data:
        question = market.get("question", "").lower()
        if query_lower in question:
            matches.append(normalize_market(market))
    
    return matches[:limit]


if __name__ == "__main__":
    # Test
    import sys
    if len(sys.argv) > 1:
        market = resolve_market(sys.argv[1])
        if market:
            print(json.dumps(market, indent=2))
        else:
            print("Market not found")
    else:
        print("Usage: python polymarket.py <market_url_or_slug>")
