"""
Correlation Patterns Database
Known relationships between market categories and specific events
"""

# Category-level correlations (general patterns)
CATEGORY_CORRELATIONS = {
    # Politics and economics
    ("politics_us", "economics"): {
        "base_correlation": 0.4,
        "notes": "US political outcomes affect Fed policy expectations",
    },
    ("politics_us", "markets"): {
        "base_correlation": 0.3,
        "notes": "Election outcomes create market uncertainty",
    },
    
    # Economics and markets
    ("economics", "markets"): {
        "base_correlation": 0.7,
        "notes": "Strong link: Fed policy → market movements",
    },
    
    # Geopolitics
    ("geopolitics", "markets"): {
        "base_correlation": 0.5,
        "notes": "War/conflict → market volatility, supply chains",
    },
    ("geopolitics", "economics"): {
        "base_correlation": 0.4,
        "notes": "Conflicts affect trade, inflation",
    },
    
    # Tech
    ("tech", "markets"): {
        "base_correlation": 0.6,
        "notes": "Big tech outcomes affect indices (heavy weighting)",
    },
}

# Specific known patterns (keyword-based)
SPECIFIC_PATTERNS = [
    {
        "trigger_keywords": ["fed", "rate cut"],
        "outcome_keywords": ["s&p", "stock", "rally"],
        "conditional_prob": 0.70,  # P(rally | rate cut)
        "inverse_prob": 0.25,      # P(rally | no rate cut)
        "confidence": "high",
        "reasoning": "Historical: Fed rate cuts boost equities 70% of time within 6 months",
    },
    {
        "trigger_keywords": ["fed", "rate hike"],
        "outcome_keywords": ["recession"],
        "conditional_prob": 0.45,
        "inverse_prob": 0.15,
        "confidence": "medium",
        "reasoning": "Rate hikes increase recession probability but not deterministic",
    },
    {
        "trigger_keywords": ["trump", "win", "president"],
        "outcome_keywords": ["republican", "senate", "congress"],
        "conditional_prob": 0.85,
        "inverse_prob": 0.35,
        "confidence": "high",
        "reasoning": "Presidential coattails: same-party congressional gains typical",
    },
    {
        "trigger_keywords": ["biden", "win", "president"],
        "outcome_keywords": ["democrat", "senate", "congress"],
        "conditional_prob": 0.80,
        "inverse_prob": 0.40,
        "confidence": "high",
        "reasoning": "Presidential coattails for Democrats",
    },
    {
        "trigger_keywords": ["war", "taiwan", "china"],
        "outcome_keywords": ["tsmc", "chip", "semiconductor"],
        "conditional_prob": 0.90,
        "inverse_prob": 0.10,
        "confidence": "high",
        "reasoning": "Taiwan conflict would devastate semiconductor supply",
    },
    {
        "trigger_keywords": ["war", "taiwan"],
        "outcome_keywords": ["recession", "crash"],
        "conditional_prob": 0.75,
        "inverse_prob": 0.20,
        "confidence": "medium",
        "reasoning": "Major geopolitical conflict → global economic disruption",
    },
    {
        "trigger_keywords": ["bitcoin", "etf", "approve"],
        "outcome_keywords": ["bitcoin", "price", "100k"],
        "conditional_prob": 0.60,
        "inverse_prob": 0.30,
        "confidence": "medium",
        "reasoning": "ETF approval historically bullish but priced in partially",
    },
    {
        "trigger_keywords": ["openai", "agi"],
        "outcome_keywords": ["ai", "regulation", "pause"],
        "conditional_prob": 0.65,
        "inverse_prob": 0.20,
        "confidence": "low",
        "reasoning": "AGI claims would likely trigger regulatory response",
    },
    {
        "trigger_keywords": ["recession", "us"],
        "outcome_keywords": ["fed", "rate cut"],
        "conditional_prob": 0.85,
        "inverse_prob": 0.30,
        "confidence": "high",
        "reasoning": "Fed typically cuts rates in response to recession",
    },
    {
        "trigger_keywords": ["inflation", "above", "3%"],
        "outcome_keywords": ["fed", "rate", "hold", "hike"],
        "conditional_prob": 0.80,
        "inverse_prob": 0.40,
        "confidence": "high",
        "reasoning": "High inflation → hawkish Fed policy",
    },
]


def get_category_correlation(cat_a: str, cat_b: str) -> dict:
    """Get base correlation between two categories."""
    # Try both orderings
    key = (cat_a, cat_b)
    if key in CATEGORY_CORRELATIONS:
        return CATEGORY_CORRELATIONS[key]
    
    key = (cat_b, cat_a)
    if key in CATEGORY_CORRELATIONS:
        return CATEGORY_CORRELATIONS[key]
    
    # Same category
    if cat_a == cat_b:
        return {"base_correlation": 0.5, "notes": "Same category - moderate correlation expected"}
    
    # Unknown
    return {"base_correlation": 0.2, "notes": "No known correlation pattern"}


def detect_mutually_exclusive(question_a: str, question_b: str) -> bool:
    """
    Detect if two markets are mutually exclusive outcomes of the same event.
    E.g., "Trump deports <250k" vs "Trump deports 250-500k" 
    """
    import re
    
    qa = question_a.lower()
    qb = question_b.lower()
    
    # Extract numbers from questions
    nums_a = set(re.findall(r'\d+', qa.replace(',', '')))
    nums_b = set(re.findall(r'\d+', qb.replace(',', '')))
    
    # If questions have numbers and they're different
    if nums_a and nums_b:
        # Remove numbers, ranges, and common filler words to get core question
        def normalize(q):
            q = re.sub(r'\d[\d,]*', '', q)  # Remove numbers
            q = re.sub(r'less than|more than|under|over|between|and|or|the|a|an|will|be|by|in|on|to', '', q)
            q = re.sub(r'[^\w\s]', '', q)  # Remove punctuation
            q = ' '.join(q.split())  # Normalize whitespace
            return q
        
        base_a = normalize(qa)
        base_b = normalize(qb)
        
        # If normalized questions are very similar, these are range buckets
        words_a = set(base_a.split())
        words_b = set(base_b.split())
        
        if not words_a or not words_b:
            return False
            
        # Check if core words match
        intersection = words_a & words_b
        union = words_a | words_b
        overlap = len(intersection) / len(union) if union else 0
        
        # Also check if one contains the other's key terms
        key_overlap = len(intersection) / min(len(words_a), len(words_b)) if min(len(words_a), len(words_b)) > 0 else 0
        
        if overlap > 0.5 or key_overlap > 0.7:
            return True
    
    return False


def find_specific_pattern(question_a: str, question_b: str) -> dict:
    """Find a specific pattern matching these two questions."""
    qa = question_a.lower()
    qb = question_b.lower()
    
    for pattern in SPECIFIC_PATTERNS:
        # Check if A triggers and B is outcome
        trigger_match = all(kw in qa for kw in pattern["trigger_keywords"])
        outcome_match = all(kw in qb for kw in pattern["outcome_keywords"])
        
        if trigger_match and outcome_match:
            return {
                "found": True,
                "direction": "a_triggers_b",
                "conditional_prob": pattern["conditional_prob"],
                "inverse_prob": pattern["inverse_prob"],
                "confidence": pattern["confidence"],
                "reasoning": pattern["reasoning"],
            }
        
        # Check reverse: B triggers A
        trigger_match = all(kw in qb for kw in pattern["trigger_keywords"])
        outcome_match = all(kw in qa for kw in pattern["outcome_keywords"])
        
        if trigger_match and outcome_match:
            return {
                "found": True,
                "direction": "b_triggers_a",
                "conditional_prob": pattern["conditional_prob"],
                "inverse_prob": pattern["inverse_prob"],
                "confidence": pattern["confidence"],
                "reasoning": pattern["reasoning"],
            }
    
    return {"found": False}


if __name__ == "__main__":
    # Test
    print("Testing pattern matching...")
    
    result = find_specific_pattern(
        "Will the Fed cut rates in Q1 2026?",
        "Will the S&P 500 hit 6000 by June 2026?"
    )
    print(f"Fed/S&P pattern: {result}")
    
    result = find_specific_pattern(
        "Will Trump win the 2028 election?",
        "Will Republicans control the Senate in 2029?"
    )
    print(f"Trump/Senate pattern: {result}")
