"""
Dashboard - Activity tracking and stats
"""

import json
import os
import time
from urllib.request import urlopen, Request
from urllib.error import URLError
from typing import Dict, Any, List
from datetime import datetime

# Configuration
PAYMENT_WALLET = os.environ.get("PAYMENT_WALLET", "0xB8B984d8150571D2Dd19aF2400D52332E262c945")
USDC_CONTRACT = "0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913"  # USDC on Base
BASESCAN_API = "https://api.basescan.org/api"
BASESCAN_KEY = os.environ.get("BASESCAN_API_KEY", "")  # Optional, rate limited without

# In-memory stats (resets on restart - could persist to file/db)
STATS = {
    "requests_total": 0,
    "requests_paid": 0,
    "requests_402": 0,
    "started_at": time.time(),
    "last_request": None,
}


def track_request(paid: bool):
    """Track an API request."""
    STATS["requests_total"] += 1
    STATS["last_request"] = time.time()
    if paid:
        STATS["requests_paid"] += 1
    else:
        STATS["requests_402"] += 1


def get_usdc_balance() -> float:
    """Get USDC balance from BaseScan API."""
    try:
        url = f"{BASESCAN_API}?module=account&action=tokenbalance&contractaddress={USDC_CONTRACT}&address={PAYMENT_WALLET}&tag=latest"
        if BASESCAN_KEY:
            url += f"&apikey={BASESCAN_KEY}"
        
        req = Request(url, headers={"User-Agent": "PolymarketAPI/1.0"})
        with urlopen(req, timeout=10) as resp:
            data = json.loads(resp.read().decode())
            if data.get("status") == "1":
                # USDC has 6 decimals
                balance = int(data.get("result", 0)) / 1e6
                return balance
    except Exception as e:
        print(f"Balance check error: {e}")
    return 0.0


def get_recent_transactions(limit: int = 10) -> List[Dict[str, Any]]:
    """Get recent USDC transfers to the wallet."""
    try:
        url = f"{BASESCAN_API}?module=account&action=tokentx&contractaddress={USDC_CONTRACT}&address={PAYMENT_WALLET}&sort=desc&page=1&offset={limit}"
        if BASESCAN_KEY:
            url += f"&apikey={BASESCAN_KEY}"
        
        req = Request(url, headers={"User-Agent": "PolymarketAPI/1.0"})
        with urlopen(req, timeout=10) as resp:
            data = json.loads(resp.read().decode())
            if data.get("status") == "1":
                txs = []
                for tx in data.get("result", []):
                    # Only show incoming transactions
                    if tx.get("to", "").lower() == PAYMENT_WALLET.lower():
                        txs.append({
                            "hash": tx.get("hash"),
                            "from": tx.get("from"),
                            "amount": int(tx.get("value", 0)) / 1e6,
                            "timestamp": datetime.fromtimestamp(int(tx.get("timeStamp", 0))).isoformat(),
                        })
                return txs
    except Exception as e:
        print(f"Transaction fetch error: {e}")
    return []


def get_dashboard_data() -> Dict[str, Any]:
    """Get full dashboard data."""
    uptime_seconds = time.time() - STATS["started_at"]
    uptime_hours = uptime_seconds / 3600
    
    balance = get_usdc_balance()
    recent_txs = get_recent_transactions(5)
    
    # Calculate revenue (from paid requests)
    price_per_request = 0.05
    estimated_revenue = STATS["requests_paid"] * price_per_request
    
    return {
        "project": "PolyEdge",
        "wallet": PAYMENT_WALLET,
        "network": "Base L2",
        "price_per_request": "$0.05 USDC",
        "stats": {
            "uptime_hours": round(uptime_hours, 2),
            "total_requests": STATS["requests_total"],
            "paid_requests": STATS["requests_paid"],
            "payment_required_responses": STATS["requests_402"],
            "conversion_rate": f"{(STATS['requests_paid'] / max(STATS['requests_total'], 1)) * 100:.1f}%",
        },
        "financials": {
            "current_balance": f"${balance:.2f} USDC",
            "estimated_session_revenue": f"${estimated_revenue:.2f}",
        },
        "recent_payments": recent_txs,
        "links": {
            "explorer": f"https://basescan.org/address/{PAYMENT_WALLET}",
            "api_docs": "https://api.nshrt.com/",
        }
    }
