"""
PolyEdge API Server
Exposes the analyzer via HTTP endpoints with x402 payment
"""

import json
import os
from http.server import HTTPServer, BaseHTTPRequestHandler
from urllib.parse import urlparse, parse_qs

# Add src to path for imports
import sys
sys.path.insert(0, os.path.dirname(__file__))

from analyzer import analyze_correlation, scan_for_opportunities
from x402 import get_payment_instructions, check_payment_header, PRICE_USDC
from dashboard import get_dashboard_data, track_request
from dashboard_html import render_dashboard_html

VERSION = "0.1.0"
PORT = int(os.environ.get("PORT", 8080))
REQUIRE_PAYMENT = os.environ.get("REQUIRE_PAYMENT", "true").lower() == "true"


class APIHandler(BaseHTTPRequestHandler):
    """Simple HTTP handler for the correlation API."""
    
    def send_json(self, data: dict, status: int = 200):
        """Send JSON response."""
        body = json.dumps(data, indent=2).encode()
        self.send_response(status)
        self.send_header("Content-Type", "application/json")
        self.send_header("Content-Length", len(body))
        self.send_header("Access-Control-Allow-Origin", "*")
        self.end_headers()
        self.wfile.write(body)
    
    def send_html(self, html: str, status: int = 200):
        """Send HTML response."""
        body = html.encode()
        self.send_response(status)
        self.send_header("Content-Type", "text/html; charset=utf-8")
        self.send_header("Content-Length", len(body))
        self.end_headers()
        self.wfile.write(body)
    
    def send_402(self):
        """Send 402 Payment Required with x402 instructions."""
        payment_info = get_payment_instructions()
        body = json.dumps(payment_info, indent=2).encode()
        self.send_response(402)
        self.send_header("Content-Type", "application/json")
        self.send_header("Content-Length", len(body))
        self.send_header("Access-Control-Allow-Origin", "*")
        self.send_header("X-402-Version", "1.0")
        self.end_headers()
        self.wfile.write(body)
    
    def get_headers_dict(self) -> dict:
        """Get request headers as dict."""
        return {k: v for k, v in self.headers.items()}
    
    def check_payment(self) -> bool:
        """Check if request has valid payment. Returns True if paid or payment not required."""
        if not REQUIRE_PAYMENT:
            return True
        
        headers = self.get_headers_dict()
        paid, msg, tx_hash = check_payment_header(headers)
        
        if paid:
            print(f"[API] Payment verified: {tx_hash[:16]}...")
            return True
        
        return False
    
    def do_OPTIONS(self):
        """Handle CORS preflight requests."""
        self.send_response(200)
        self.send_header("Access-Control-Allow-Origin", "*")
        self.send_header("Access-Control-Allow-Methods", "GET, OPTIONS")
        self.send_header("Access-Control-Allow-Headers", "X-Payment, X-402-Payment, Authorization, Content-Type")
        self.send_header("Access-Control-Max-Age", "86400")
        self.end_headers()
    
    def do_GET(self):
        """Handle GET requests."""
        parsed = urlparse(self.path)
        path = parsed.path
        params = parse_qs(parsed.query)
        
        # Health check
        if path == "/health":
            self.send_json({"status": "ok", "version": "0.1.0"})
            return
        
        # Dashboard - human-friendly HTML version
        if path == "/dashboard":
            html = render_dashboard_html()
            self.send_html(html)
            return
        
        # Dashboard JSON - for programmatic access
        if path == "/api/v1/dashboard":
            dashboard = get_dashboard_data()
            self.send_json(dashboard)
            return
        
        # API info
        if path == "/" or path == "/api":
            self.send_json({
                "name": "PolyEdge",
                "version": "0.1.0",
                "pricing": {
                    "correlation": f"${PRICE_USDC} USDC per request",
                    "network": "Base L2",
                    "protocol": "x402"
                },
                "endpoints": {
                    "/health": "Health check (free)",
                    "/dashboard": "Activity dashboard and stats (free)",
                    "/api/v1/correlation": "Analyze correlation between two markets (paid)",
                },
                "usage": {
                    "1": "GET /api/v1/correlation?a=<slug>&b=<slug>",
                    "2": "Receive 402 Payment Required with payment instructions",
                    "3": "Send USDC payment to specified address",
                    "4": "Retry with X-Payment: <tx_hash> header"
                },
                "example": "/api/v1/correlation?a=fed-rate-cut&b=sp500-6000"
            })
            return
        
        # Correlation analysis (PAID endpoint)
        if path == "/api/v1/correlation":
            # Check payment first
            if not self.check_payment():
                track_request(paid=False)
                self.send_402()
                return
            
            track_request(paid=True)
            
            market_a = params.get("a", [None])[0]
            market_b = params.get("b", [None])[0]
            
            if not market_a or not market_b:
                self.send_json({
                    "error": "Missing parameters",
                    "usage": "/api/v1/correlation?a=<market_a>&b=<market_b>",
                    "example": "/api/v1/correlation?a=fed-rate-cut&b=sp500-6000"
                }, status=400)
                return
            
            try:
                result = analyze_correlation(market_a, market_b)
                if "error" in result:
                    self.send_json(result, status=404)
                else:
                    self.send_json(result)
            except Exception as e:
                self.send_json({"error": str(e)}, status=500)
            return
        
        # 404
        self.send_json({"error": "Not found", "path": path}, status=404)
    
    def log_message(self, format, *args):
        """Log requests."""
        print(f"[API] {args[0]}")


def main():
    """Run the API server."""
    server = HTTPServer(("0.0.0.0", PORT), APIHandler)
    print(f"PolyEdge API running on port {PORT}")
    server.serve_forever()


if __name__ == "__main__":
    main()
