"""
Polymarket Correlation Analyzer
Detects mispriced correlations between prediction markets
"""

import json
from typing import Dict, Any, Optional, Tuple

from polymarket import resolve_market, normalize_market
from patterns import get_category_correlation, find_specific_pattern, detect_mutually_exclusive


def analyze_correlation(market_a_id: str, market_b_id: str) -> Dict[str, Any]:
    """
    Analyze correlation between two markets.
    Returns mispricing signal if found.
    """
    # Fetch both markets
    market_a = resolve_market(market_a_id)
    market_b = resolve_market(market_b_id)
    
    if not market_a:
        return {"error": f"Could not find market A: {market_a_id}"}
    if not market_b:
        return {"error": f"Could not find market B: {market_b_id}"}
    
    # Check if markets are active
    if market_a.get("closed"):
        return {"error": f"Market A is closed: {market_a['question'][:50]}"}
    if market_b.get("closed"):
        return {"error": f"Market B is closed: {market_b['question'][:50]}"}
    
    # Get prices
    p_a = market_a["yes_price"]
    p_b = market_b["yes_price"]
    
    # Check for mutually exclusive markets first
    if detect_mutually_exclusive(market_a["question"], market_b["question"]):
        # These are buckets of the same event - negatively correlated
        # Sum of YES prices should approximate 1.0 across all buckets
        return {
            "market_a": {
                "id": market_a["id"],
                "question": market_a["question"],
                "yes_price": round(p_a, 4),
                "category": market_a["category"],
                "volume": market_a["volume"],
            },
            "market_b": {
                "id": market_b["id"],
                "question": market_b["question"],
                "yes_price": round(p_b, 4),
                "category": market_b["category"],
                "volume": market_b["volume"],
            },
            "analysis": {
                "pattern_type": "mutually_exclusive",
                "relationship": "These markets are outcome buckets of the same event",
                "combined_probability": round(p_a + p_b, 4),
                "notes": "Mutually exclusive outcomes should not be analyzed for correlation. Consider comparing to other independent events.",
                "confidence": "high",
            },
            "signal": {
                "action": "SKIP",
                "strength": "none",
                "reason": "Markets are mutually exclusive - not suitable for correlation analysis",
            },
        }
    
    # Find correlation pattern
    specific = find_specific_pattern(market_a["question"], market_b["question"])
    category_corr = get_category_correlation(market_a["category"], market_b["category"])
    
    # Calculate expected price
    if specific["found"]:
        # Use specific pattern
        if specific["direction"] == "a_triggers_b":
            # P(B) = P(A) * P(B|A) + P(¬A) * P(B|¬A)
            p_b_given_a = specific["conditional_prob"]
            p_b_given_not_a = specific["inverse_prob"]
            expected_p_b = (p_a * p_b_given_a) + ((1 - p_a) * p_b_given_not_a)
            
            mispricing = expected_p_b - p_b
            analysis = {
                "pattern_type": "specific",
                "direction": "a_triggers_b",
                "p_b_given_a": p_b_given_a,
                "p_b_given_not_a": p_b_given_not_a,
                "expected_price_b": round(expected_p_b, 4),
                "actual_price_b": round(p_b, 4),
                "mispricing": round(mispricing, 4),
                "confidence": specific["confidence"],
                "reasoning": specific["reasoning"],
            }
        else:
            # B triggers A
            p_a_given_b = specific["conditional_prob"]
            p_a_given_not_b = specific["inverse_prob"]
            expected_p_a = (p_b * p_a_given_b) + ((1 - p_b) * p_a_given_not_b)
            
            mispricing = expected_p_a - p_a
            analysis = {
                "pattern_type": "specific",
                "direction": "b_triggers_a",
                "p_a_given_b": p_a_given_b,
                "p_a_given_not_b": p_a_given_not_b,
                "expected_price_a": round(expected_p_a, 4),
                "actual_price_a": round(p_a, 4),
                "mispricing": round(mispricing, 4),
                "confidence": specific["confidence"],
                "reasoning": specific["reasoning"],
            }
    else:
        # Use category-level correlation
        base_corr = category_corr["base_correlation"]
        
        # Simple correlation model: if categories correlate, prices should track
        # Expected: if A is high, B should be pulled toward high (and vice versa)
        # This is a simplified model - real correlation analysis would need historical data
        
        # Estimate expected B based on A's price and correlation
        neutral = 0.5
        expected_p_b = neutral + base_corr * (p_a - neutral)
        mispricing = expected_p_b - p_b
        
        analysis = {
            "pattern_type": "category",
            "categories": [market_a["category"], market_b["category"]],
            "base_correlation": base_corr,
            "expected_price_b": round(expected_p_b, 4),
            "actual_price_b": round(p_b, 4),
            "mispricing": round(mispricing, 4),
            "confidence": "low",
            "reasoning": category_corr["notes"],
        }
    
    # Generate signal
    signal = generate_signal(analysis)
    
    return {
        "market_a": {
            "id": market_a["id"],
            "question": market_a["question"],
            "yes_price": round(p_a, 4),
            "category": market_a["category"],
            "volume": market_a["volume"],
        },
        "market_b": {
            "id": market_b["id"],
            "question": market_b["question"],
            "yes_price": round(p_b, 4),
            "category": market_b["category"],
            "volume": market_b["volume"],
        },
        "analysis": analysis,
        "signal": signal,
    }


def generate_signal(analysis: Dict[str, Any]) -> Dict[str, Any]:
    """Generate trading signal from analysis."""
    mispricing = analysis.get("mispricing", 0)
    confidence = analysis.get("confidence", "low")
    
    # Thresholds
    if confidence == "high":
        threshold = 0.05  # 5%
    elif confidence == "medium":
        threshold = 0.08  # 8%
    else:
        threshold = 0.12  # 12%
    
    abs_mispricing = abs(mispricing)
    
    if abs_mispricing < threshold:
        return {
            "action": "HOLD",
            "strength": "none",
            "reason": f"Mispricing ({abs_mispricing:.1%}) below threshold ({threshold:.1%})",
        }
    
    # Determine direction
    direction = analysis.get("direction", "a_triggers_b")
    
    if direction == "a_triggers_b" or analysis.get("pattern_type") == "category":
        # Market B is the one to trade
        if mispricing > 0:
            action = "BUY_YES_B"
            reason = f"Market B underpriced by {mispricing:.1%}"
        else:
            action = "BUY_NO_B"
            reason = f"Market B overpriced by {abs_mispricing:.1%}"
    else:
        # Market A is the one to trade
        if mispricing > 0:
            action = "BUY_YES_A"
            reason = f"Market A underpriced by {mispricing:.1%}"
        else:
            action = "BUY_NO_A"
            reason = f"Market A overpriced by {abs_mispricing:.1%}"
    
    # Strength
    if abs_mispricing > threshold * 2:
        strength = "strong"
    else:
        strength = "moderate"
    
    return {
        "action": action,
        "strength": strength,
        "reason": reason,
        "mispricing_pct": f"{abs_mispricing:.1%}",
        "confidence": confidence,
    }


def scan_for_opportunities(markets: list, min_volume: float = 100000) -> list:
    """
    Scan a list of markets for correlation opportunities.
    Returns list of potential mispricings.
    """
    opportunities = []
    
    # Filter by volume
    filtered = [m for m in markets if m.get("volume", 0) >= min_volume]
    
    # Compare all pairs (O(n²) but markets list is small)
    for i, market_a in enumerate(filtered):
        for market_b in filtered[i+1:]:
            result = analyze_correlation(
                market_a.get("slug") or market_a.get("id"),
                market_b.get("slug") or market_b.get("id")
            )
            
            if "error" not in result:
                signal = result.get("signal", {})
                if signal.get("action") != "HOLD":
                    opportunities.append(result)
    
    # Sort by mispricing magnitude
    opportunities.sort(
        key=lambda x: abs(x.get("analysis", {}).get("mispricing", 0)),
        reverse=True
    )
    
    return opportunities


if __name__ == "__main__":
    import sys
    
    if len(sys.argv) == 3:
        result = analyze_correlation(sys.argv[1], sys.argv[2])
        print(json.dumps(result, indent=2))
    else:
        print("Usage: python analyzer.py <market_a> <market_b>")
        print("Example: python analyzer.py 'fed-rate-cut' 'sp500-6000'")
