#!/usr/bin/env node
// podman-browse - Headless browser automation using Podman + Playwright
// Fetches JavaScript-rendered pages and returns text or HTML content

const { spawn } = require('child_process');
const path = require('path');

const IMAGE = 'mcr.microsoft.com/playwright:v1.50.0-noble';
const PLAYWRIGHT_VERSION = '1.50.0';

// Defaults
let waitMs = 2000;
let outputMode = 'text';
let selector = '';
let url = '';

// Parse arguments
const args = process.argv.slice(2);
for (let i = 0; i < args.length; i++) {
    switch (args[i]) {
        case '--html':
            outputMode = 'html';
            break;
        case '--wait':
            waitMs = parseInt(args[++i], 10);
            break;
        case '--selector':
            selector = args[++i];
            break;
        case '-h':
        case '--help':
            console.log(`Usage: browse.js [OPTIONS] URL

Fetch a JavaScript-rendered page using Playwright in Podman.

Options:
  --html              Return raw HTML instead of text
  --wait <ms>         Wait time after load (default: 2000ms)
  --selector <css>    Wait for specific element before capturing
  -h, --help          Show this help

Examples:
  ./browse.js "https://example.com"
  ./browse.js --html "https://example.com"
  ./browse.js --selector ".content" --wait 5000 "https://example.com"`);
            process.exit(0);
        default:
            if (!args[i].startsWith('-')) {
                url = args[i];
            } else {
                console.error(`Unknown option: ${args[i]}`);
                process.exit(1);
            }
    }
}

if (!url) {
    console.error('Error: URL is required');
    console.error('Usage: browse.js [OPTIONS] URL');
    process.exit(1);
}

// The actual Playwright script to run inside the container
const playwrightScript = `
const { chromium } = require('playwright');

(async () => {
    const url = process.env.TARGET_URL;
    const waitMs = parseInt(process.env.WAIT_MS || '2000', 10);
    const outputMode = process.env.OUTPUT_MODE || 'text';
    const selector = process.env.SELECTOR || '';

    const browser = await chromium.launch({
        headless: true,
        args: ['--no-sandbox', '--disable-setuid-sandbox']
    });

    try {
        const context = await browser.newContext({
            userAgent: 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
        });
        const page = await context.newPage();

        await page.goto(url, { waitUntil: 'networkidle', timeout: 30000 });

        if (selector) {
            try {
                await page.waitForSelector(selector, { timeout: 10000 });
            } catch (e) {
                console.error('Warning: selector not found within timeout');
            }
        }

        await page.waitForTimeout(waitMs);

        if (outputMode === 'html') {
            const html = await page.content();
            console.log(html);
        } else {
            const text = await page.evaluate(() => {
                const scripts = document.querySelectorAll('script, style, noscript');
                scripts.forEach(el => el.remove());
                return document.body.innerText;
            });
            console.log(text);
        }
    } catch (error) {
        console.error('Error:', error.message);
        process.exit(1);
    } finally {
        await browser.close();
    }
})();
`;

// Run Playwright in Podman container
const podmanArgs = [
    'run', '--rm', '-i',
    '--ipc=host',
    '--init',
    '-e', `TARGET_URL=${url}`,
    '-e', `WAIT_MS=${waitMs}`,
    '-e', `OUTPUT_MODE=${outputMode}`,
    '-e', `SELECTOR=${selector}`,
    IMAGE,
    '/bin/bash', '-c',
    `cd /tmp && npm init -y >/dev/null 2>&1 && npm install playwright@${PLAYWRIGHT_VERSION} >/dev/null 2>&1 && node -e '${playwrightScript.replace(/'/g, "'\\''")}'`
];

const proc = spawn('podman', podmanArgs, { stdio: ['inherit', 'inherit', 'inherit'] });

proc.on('close', (code) => {
    process.exit(code || 0);
});

proc.on('error', (err) => {
    console.error(`Failed to start podman: ${err.message}`);
    process.exit(1);
});
