---
name: podio
description: |
  Podio API integration with managed OAuth. Manage workspaces, apps, items, tasks, and comments.
  Use this skill when users want to read, create, update, or delete Podio items, manage tasks, or interact with Podio apps and workspaces.
  For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
compatibility: Requires network access and valid Maton API key
metadata:
  author: maton
  version: "1.0"
  clawdbot:
    emoji: 🧠
    homepage: "https://maton.ai"
    requires:
      env:
        - MATON_API_KEY
---

# Podio

Access the Podio API with managed OAuth authentication. Manage organizations, workspaces (spaces), apps, items, tasks, comments, and files.

## Quick Start

```bash
# List organizations
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/podio/org/')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

## Base URL

```
https://gateway.maton.ai/podio/{native-api-path}
```

Replace `{native-api-path}` with the actual Podio API endpoint path. The gateway proxies requests to `api.podio.com` and automatically injects your OAuth token.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your Podio OAuth connections at `https://ctrl.maton.ai`.

### List Connections

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections?app=podio&status=ACTIVE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Create Connection

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'app': 'podio'}).encode()
req = urllib.request.Request('https://ctrl.maton.ai/connections', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Get Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

**Response:**
```json
{
  "connection": {
    "connection_id": "21fd90f9-5935-43cd-b6c8-bde9d915ca80",
    "status": "ACTIVE",
    "creation_time": "2025-12-08T07:20:53.488460Z",
    "last_updated_time": "2026-01-31T20:03:32.593153Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "podio",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}', method='DELETE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Specifying Connection

If you have multiple Podio connections, specify which one to use with the `Maton-Connection` header:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/podio/org/')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Maton-Connection', '21fd90f9-5935-43cd-b6c8-bde9d915ca80')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

If omitted, the gateway uses the default (oldest) active connection.

## API Reference

### Organization Operations

#### List Organizations

Returns all organizations and spaces the user is a member of.

```bash
GET /podio/org/
```

**Response:**
```json
[
  {
    "org_id": 123456,
    "name": "My Organization",
    "url": "https://podio.com/myorg",
    "url_label": "myorg",
    "type": "premium",
    "role": "admin",
    "status": "active",
    "spaces": [
      {
        "space_id": 789,
        "name": "Project Space",
        "url": "https://podio.com/myorg/project-space",
        "role": "admin"
      }
    ]
  }
]
```

#### Get Organization

```bash
GET /podio/org/{org_id}
```

### Space (Workspace) Operations

#### Get Space

```bash
GET /podio/space/{space_id}
```

**Response:**
```json
{
  "space_id": 789,
  "name": "Project Space",
  "privacy": "closed",
  "auto_join": false,
  "url": "https://podio.com/myorg/project-space",
  "url_label": "project-space",
  "role": "admin",
  "created_on": "2025-01-15T10:30:00Z",
  "created_by": {
    "user_id": 12345,
    "name": "John Doe"
  }
}
```

#### Create Space

```bash
POST /podio/space/
Content-Type: application/json

{
  "org_id": 123456,
  "name": "New Project Space",
  "privacy": "closed",
  "auto_join": false,
  "post_on_new_app": true,
  "post_on_new_member": true
}
```

**Response:**
```json
{
  "space_id": 790,
  "url": "https://podio.com/myorg/new-project-space"
}
```

### Application Operations

#### Get Apps by Space

```bash
GET /podio/app/space/{space_id}/
```

Optional query parameters:
- `include_inactive` - Include inactive apps (default: false)

#### Get App

```bash
GET /podio/app/{app_id}
```

**Response:**
```json
{
  "app_id": 456,
  "status": "active",
  "space_id": 789,
  "config": {
    "name": "Tasks",
    "item_name": "Task",
    "description": "Track project tasks",
    "icon": "list"
  },
  "fields": [...]
}
```

### Item Operations

#### Get Item

```bash
GET /podio/item/{item_id}
```

Optional query parameters:
- `mark_as_viewed` - Mark notifications as viewed (default: true)

**Response:**
```json
{
  "item_id": 123,
  "title": "Complete project plan",
  "app": {
    "app_id": 456,
    "name": "Tasks"
  },
  "fields": [
    {
      "field_id": 1,
      "external_id": "status",
      "type": "category",
      "values": [{"value": {"text": "In Progress"}}]
    }
  ],
  "created_on": "2025-01-20T14:00:00Z",
  "created_by": {
    "user_id": 12345,
    "name": "John Doe"
  }
}
```

#### Filter Items

```bash
POST /podio/item/app/{app_id}/filter/
Content-Type: application/json

{
  "sort_by": "created_on",
  "sort_desc": true,
  "filters": {
    "status": [1, 2]
  },
  "limit": 30,
  "offset": 0
}
```

**Response:**
```json
{
  "total": 150,
  "filtered": 45,
  "items": [
    {
      "item_id": 123,
      "title": "Complete project plan",
      "fields": [...],
      "comment_count": 5,
      "file_count": 2
    }
  ]
}
```

#### Add New Item

```bash
POST /podio/item/app/{app_id}/
Content-Type: application/json

{
  "fields": {
    "title": "New task",
    "status": 1,
    "due-date": {"start": "2025-02-15"}
  },
  "tags": ["urgent", "project-alpha"],
  "file_ids": [12345]
}
```

Optional query parameters:
- `hook` - Execute hooks (default: true)
- `silent` - Suppress notifications (default: false)

**Response:**
```json
{
  "item_id": 124,
  "title": "New task"
}
```

#### Update Item

```bash
PUT /podio/item/{item_id}
Content-Type: application/json

{
  "fields": {
    "status": 2
  },
  "revision": 5
}
```

Optional query parameters:
- `hook` - Execute hooks (default: true)
- `silent` - Suppress notifications (default: false)

**Response:**
```json
{
  "revision": 6,
  "title": "New task"
}
```

#### Delete Item

```bash
DELETE /podio/item/{item_id}
```

Optional query parameters:
- `hook` - Execute hooks (default: true)
- `silent` - Suppress notifications (default: false)

### Task Operations

#### Get Tasks

**Note:** Tasks require at least one filter: `org`, `space`, `app`, `responsible`, `reference`, `created_by`, or `completed_by`.

```bash
GET /podio/task/?org={org_id}
GET /podio/task/?space={space_id}
GET /podio/task/?app={app_id}&completed=false
```

Query parameters:
- `org` - Filter by organization ID (required if no other filter)
- `space` - Filter by space ID
- `app` - Filter by app ID
- `completed` - Filter by completion status (`true` or `false`)
- `responsible` - Filter by responsible user IDs
- `created_by` - Filter by creator
- `due_date` - Date range (YYYY-MM-DD-YYYY-MM-DD)
- `limit` - Maximum results
- `offset` - Result offset
- `sort_by` - Sort by: created_on, completed_on, rank (default: rank)
- `grouping` - Group by: due_date, created_by, responsible, app, space, org

#### Get Task

```bash
GET /podio/task/{task_id}
```

**Response:**
```json
{
  "task_id": 789,
  "text": "Review project proposal",
  "description": "Detailed review of the Q1 proposal",
  "status": "active",
  "due_date": "2025-02-15",
  "due_time": "17:00:00",
  "responsible": {
    "user_id": 12345,
    "name": "John Doe"
  },
  "created_on": "2025-01-20T10:00:00Z",
  "labels": [
    {"label_id": 1, "text": "High Priority", "color": "red"}
  ]
}
```

#### Create Task

```bash
POST /podio/task/
Content-Type: application/json

{
  "text": "Review project proposal",
  "description": "Detailed review of the Q1 proposal",
  "due_date": "2025-02-15",
  "due_time": "17:00:00",
  "responsible": 12345,
  "private": false,
  "ref_type": "item",
  "ref_id": 123,
  "labels": [1, 2]
}
```

Optional query parameters:
- `hook` - Execute hooks (default: true)
- `silent` - Suppress notifications (default: false)

**Response:**
```json
{
  "task_id": 790,
  ...
}
```

### Comment Operations

#### Get Comments on Object

```bash
GET /podio/comment/{type}/{id}/
```

Where `{type}` is the object type (e.g., "item", "task") and `{id}` is the object ID.

Optional query parameters:
- `limit` - Maximum comments (default: 100)
- `offset` - Pagination offset (default: 0)

**Response:**
```json
[
  {
    "comment_id": 456,
    "value": "This looks great!",
    "created_on": "2025-01-20T15:30:00Z",
    "created_by": {
      "user_id": 12345,
      "name": "John Doe"
    },
    "files": []
  }
]
```

#### Add Comment to Object

```bash
POST /podio/comment/{type}/{id}
Content-Type: application/json

{
  "value": "Great progress on this task!",
  "file_ids": [12345],
  "embed_url": "https://example.com/doc"
}
```

Optional query parameters:
- `alert_invite` - Auto-invite mentioned users (default: false)
- `hook` - Execute hooks (default: true)
- `silent` - Suppress notifications (default: false)

**Response:**
```json
{
  "comment_id": 457,
  ...
}
```

## Pagination

Podio uses offset-based pagination with `limit` and `offset` parameters:

```bash
POST /podio/item/app/{app_id}/filter/
Content-Type: application/json

{
  "limit": 30,
  "offset": 0
}
```

Response includes total counts:
```json
{
  "total": 150,
  "filtered": 45,
  "items": [...]
}
```

For subsequent pages, increment the offset:
```json
{
  "limit": 30,
  "offset": 30
}
```

## Code Examples

### JavaScript

```javascript
const response = await fetch(
  'https://gateway.maton.ai/podio/org/',
  {
    headers: {
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    }
  }
);
const data = await response.json();
```

### Python

```python
import os
import requests

response = requests.get(
    'https://gateway.maton.ai/podio/org/',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'}
)
data = response.json()
```

## Notes

- Organization IDs, space IDs, app IDs, and item IDs are integers
- Field values can be specified by field_id or external_id
- Category fields use option IDs (integers), not text values
- Deleting an item also deletes associated tasks (cascade delete)
- Tasks require at least one filter (org, space, app, responsible, reference, created_by, or completed_by)
- Use `silent=true` to suppress notifications for bulk operations
- Use `hook=false` to skip webhook triggers
- Include `revision` in update requests for conflict detection (returns 409 if conflict)
- IMPORTANT: When using curl commands, use `curl -g` when URLs contain brackets to disable glob parsing
- IMPORTANT: When piping curl output to `jq` or other commands, environment variables like `$MATON_API_KEY` may not expand correctly in some shell environments

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Missing Podio connection or invalid request |
| 401 | Invalid or missing Maton API key |
| 403 | Forbidden - insufficient permissions |
| 404 | Resource not found |
| 409 | Conflict (revision mismatch on update) |
| 410 | Resource has been deleted |
| 429 | Rate limited |
| 4xx/5xx | Passthrough error from Podio API |

### Troubleshooting: API Key Issues

1. Check that the `MATON_API_KEY` environment variable is set:

```bash
echo $MATON_API_KEY
```

2. Verify the API key is valid by listing connections:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Troubleshooting: Invalid App Name

1. Ensure your URL path starts with `podio`. For example:

- Correct: `https://gateway.maton.ai/podio/org/`
- Incorrect: `https://gateway.maton.ai/org/`

## Resources

- [Podio API Documentation](https://developers.podio.com/doc)
- [Podio API Authentication](https://developers.podio.com/authentication)
- [Podio Items API](https://developers.podio.com/doc/items)
- [Podio Tasks API](https://developers.podio.com/doc/tasks)
- [Maton Community](https://discord.com/invite/dBfFAcefs2)
- [Maton Support](mailto:support@maton.ai)
