---
name: pocketsmith
description: Manage PocketSmith transactions, categories, and financial data via the API
metadata: {"openclaw": {"category": "finance", "requires": {"env": ["POCKETSMITH_DEVELOPER_KEY"]}, "optional_env": ["POCKETSMITH_ALLOW_WRITES"]}}
---

# PocketSmith Skill

Manage PocketSmith transactions and categories. Supports listing, searching, creating, updating, and deleting financial data.

## Prerequisites

Set these environment variables:
- `POCKETSMITH_DEVELOPER_KEY` - Your PocketSmith developer key (from Settings > Security > Manage Developer Keys)
- `POCKETSMITH_ALLOW_WRITES` - Set to `true` to enable create, update, and delete operations (disabled by default for safety)

## Commands

All commands should be run from the skill directory using `uv run pocketsmith`.

### Authentication

```bash
# Check authentication status and get user info
pocketsmith auth status

# Get current user details
pocketsmith me
```

### Transactions

```bash
# Get a single transaction
pocketsmith transactions get <transaction_id>

# List transactions for a user
pocketsmith transactions list-by-user <user_id>
pocketsmith transactions list-by-user <user_id> --start-date 2024-01-01 --end-date 2024-12-31
pocketsmith transactions list-by-user <user_id> --search "coffee"
pocketsmith transactions list-by-user <user_id> --uncategorised
pocketsmith transactions list-by-user <user_id> --needs-review
pocketsmith transactions list-by-user <user_id> --type debit

# List transactions by account
pocketsmith transactions list-by-account <account_id>

# List transactions by category
pocketsmith transactions list-by-category <category_ids>  # comma-separated

# List transactions by transaction account
pocketsmith transactions list-by-transaction-account <transaction_account_id>

# Create a transaction (requires POCKETSMITH_ALLOW_WRITES=true)
pocketsmith transactions create <transaction_account_id> --payee "Store Name" --amount -50.00 --date 2024-01-15
pocketsmith transactions create <transaction_account_id> --payee "Salary" --amount 5000.00 --date 2024-01-01 --category-id 123

# Update a transaction (requires POCKETSMITH_ALLOW_WRITES=true)
pocketsmith transactions update <transaction_id> --category-id 456
pocketsmith transactions update <transaction_id> --payee "New Payee" --note "Updated note"
pocketsmith transactions update <transaction_id> --labels "groceries,essential"

# Delete a transaction (requires POCKETSMITH_ALLOW_WRITES=true)
pocketsmith transactions delete <transaction_id>
```

### Categories

```bash
# Get a single category
pocketsmith categories get <category_id>

# List all categories for a user
pocketsmith categories list <user_id>

# Create a category (requires POCKETSMITH_ALLOW_WRITES=true)
pocketsmith categories create <user_id> --title "New Category"
pocketsmith categories create <user_id> --title "Subcategory" --parent-id 123
pocketsmith categories create <user_id> --title "Bills" --colour "#ff0000" --is-bill true

# Update a category (requires POCKETSMITH_ALLOW_WRITES=true)
pocketsmith categories update <category_id> --title "Renamed Category"
pocketsmith categories update <category_id> --parent-id 456
pocketsmith categories update <category_id> --no-parent  # Make top-level
pocketsmith categories update <category_id> --colour "#00ff00"

# Delete a category (requires POCKETSMITH_ALLOW_WRITES=true)
pocketsmith categories delete <category_id>
```

### Labels

```bash
# List all labels for a user
pocketsmith labels list <user_id>
```

### Budget

```bash
# List budget for a user (per-category budget analysis)
pocketsmith budget list <user_id>
pocketsmith budget list <user_id> --roll-up true

# Get budget summary for a user
pocketsmith budget summary <user_id> --period months --interval 1 --start-date 2024-01-01 --end-date 2024-12-31

# Get trend analysis (requires category and scenario IDs)
pocketsmith budget trend <user_id> --period months --interval 1 --start-date 2024-01-01 --end-date 2024-12-31 --categories "123,456" --scenarios "1,2"

# Refresh forecast cache (requires POCKETSMITH_ALLOW_WRITES=true)
pocketsmith budget refresh <user_id>
```

## Transaction Filter Options

When listing transactions, these filters are available:
- `--start-date` - Filter from date (YYYY-MM-DD)
- `--end-date` - Filter to date (YYYY-MM-DD)
- `--updated-since` - Only transactions updated after this datetime
- `--uncategorised` - Only uncategorised transactions
- `--type` - Filter by type: `debit` or `credit`
- `--needs-review` - Only transactions needing review
- `--search` - Search term for payee/memo
- `--page` - Page number for pagination

## Category Options

When creating/updating categories:
- `--title` - Category name
- `--colour` - CSS hex colour (e.g., `#ff0000`)
- `--parent-id` - Parent category ID for subcategories
- `--no-parent` - Make category top-level (update only)
- `--is-transfer` - Mark as transfer category (true/false)
- `--is-bill` - Mark as bill category (true/false)
- `--roll-up` - Roll up to parent category (true/false)
- `--refund-behaviour` - `debit_only` or `credit_only`

## Output Format

All commands output JSON. Example transaction:

```json
{
  "id": 1234567,
  "payee": "Coffee Shop",
  "amount": -5.50,
  "date": "2024-01-15",
  "category": {
    "id": 123,
    "title": "Eating Out"
  },
  "transaction_account": {
    "id": 456,
    "name": "Checking Account"
  }
}
```

## Date Format

All dates use `YYYY-MM-DD` format (e.g., `2024-01-15`).

## Write Protection

Write operations (create, update, delete) are **disabled by default** for safety. To enable them:

```bash
export POCKETSMITH_ALLOW_WRITES=true
```

Without this, write commands will fail with:

```json
{"error": "Write operations are disabled by default. Set POCKETSMITH_ALLOW_WRITES=true to enable create, update, and delete operations.", "hint": "export POCKETSMITH_ALLOW_WRITES=true"}
```

## Common Workflows

### Search and Categorize Transactions

```bash
# Find uncategorised transactions
pocketsmith transactions list-by-user 123456 --uncategorised

# Search for specific transactions
pocketsmith transactions list-by-user 123456 --search "Netflix"

# Categorize a transaction
pocketsmith transactions update 789012 --category-id 456
```

### Organize Categories

```bash
# List existing categories
pocketsmith categories list 123456

# Create a new subcategory
pocketsmith categories create 123456 --title "Streaming" --parent-id 789

# Move a category under a different parent
pocketsmith categories update 101112 --parent-id 789
```

### Review Transactions

```bash
# Find transactions needing review
pocketsmith transactions list-by-user 123456 --needs-review

# Mark as reviewed by updating
pocketsmith transactions update 789012 --needs-review false
```
