---
name: agent-tools
description: Digital Swiss Army knife for everyday labor that standard models can't handle out of the box. Use when: (1) Need to manipulate files (rename, move, copy, delete), (2) Working with JSON/YAML/TOML configs, (3) Running system commands safely, (4) Processing text with regex or transformations, (5) Need utility functions for common operations.
---

# Agent Tools - Universal Utility Belt

A collection of practical utilities for everyday agent operations.

## File Operations

### Safe File Manipulation

Always use `trash` instead of `rm` when possible:

```bash
trash /path/to/file  # Safer deletion (recoverable)
```

### Bulk File Operations

```bash
# Rename files with pattern
for f in *.txt; do mv "$f" "${f/.txt/.md}"; done

# Find and delete files older than 7 days
find . -name "*.log" -mtime +7 -exec trash {} \;

# Copy with progress
rsync -av --progress src/ dest/
```

## JSON/YAML Processing

### JSON Operations (jq)

```bash
# Pretty print
jq '.' file.json

# Extract field
jq '.field' file.json

# Update field
jq '.field = "new_value"' file.json > tmp && mv tmp file.json

# Merge JSON files
jq -s 'add' file1.json file2.json
```

### YAML Operations (yq)

```bash
# Read value
yq '.key' file.yaml

# Update value
yq '.key = "value"' -i file.yaml

# Convert YAML to JSON
yq -o=json '.' file.yaml
```

## Text Processing

### Common Patterns

```bash
# Search and replace in files
sed -i '' 's/old/new/g' file.txt

# Extract matches
grep -oP 'pattern' file.txt

# Count occurrences
grep -c 'pattern' file.txt

# Remove duplicate lines
sort file.txt | uniq > deduplicated.txt

# Extract column
awk '{print $2}' file.txt
```

## System Utilities

### Process Management

```bash
# Find process by name
ps aux | grep process_name

# Kill by port
lsof -ti:3000 | xargs kill -9

# Monitor resource usage
htop
```

### Network Operations

```bash
# Check port availability
lsof -i :PORT

# Download with retry
curl --retry 3 -O URL

# Test endpoint
curl -X POST -H "Content-Type: application/json" -d '{"key":"value"}' URL
```

## Encoding/Decoding

```bash
# Base64 encode/decode
echo "text" | base64
echo "dGV4dAo=" | base64 -d

# URL encode/decode
python3 -c "import urllib.parse; print(urllib.parse.quote('text'))"
python3 -c "import urllib.parse; print(urllib.parse.unquote('text%20here'))"

# JSON escape/unescape
jq -R . <<< 'string to escape'
jq -r . <<< '"escaped string"'
```

## Date/Time Utilities

```bash
# Current timestamp
date +%s

# ISO format
date -u +"%Y-%m-%dT%H:%M:%SZ"

# Convert timestamp
date -r 1234567890

# Timezone conversion
TZ="America/Chicago" date
```

## Validation Helpers

```bash
# Validate JSON
jq empty file.json && echo "Valid JSON"

# Validate YAML
python3 -c "import yaml; yaml.safe_load(open('file.yaml'))" && echo "Valid YAML"

# Check JSON schema
check-jsonschema --schemafile schema.json document.json
```

## Quick Reference

| Task | Command |
|------|---------|
| Safe delete | `trash file` |
| Find files | `find . -name "*.ext"` |
| Search in files | `grep -r "pattern" .` |
| Replace text | `sed -i '' 's/old/new/g'` |
| JSON pretty | `jq '.'` |
| YAML read | `yq '.key'` |
| Port check | `lsof -i :PORT` |
| Base64 decode | `base64 -d` |
