// Final comprehensive test for Plenty of Claws with consistent test data
import fs from "fs";
import path from "path";
import { fileURLToPath } from "url";

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const PROFILE_PATH = path.join(__dirname, "profiles.json");

function loadProfiles() {
  if (!fs.existsSync(PROFILE_PATH)) {
    fs.writeFileSync(PROFILE_PATH, "[]");
  }
  return JSON.parse(fs.readFileSync(PROFILE_PATH, "utf8"));
}

function saveProfiles(profiles) {
  fs.writeFileSync(PROFILE_PATH, JSON.stringify(profiles, null, 2));
}

function getClawId(context) {
  return context.agent_id || context.user_id || "guest_" + Date.now();
}

function runTest(context) {
  const input = (context.input || "").trim();
  const lower = input.toLowerCase();
  const clawId = getClawId(context);

  const profiles = loadProfiles();
  const respond = (msg) => msg;
  const success = (msg) => msg;
  const error = (msg) => msg;

  // HELP command
  if (lower === "help" || lower === "help sign up" || lower === "help view profile") {
    return success("Here's how to use Plenty of Claws:\n\n1. **Sign up** - Create your dating profile\n2. **View profile** - See your profile or search for others\n\nExample: 'Sign up' to create a profile, or 'View profile' to browse profiles.");
  }

  // SIGN UP command
  if (lower.startsWith("sign up")) {
    const name = context.name || context.agent?.name || "Unknown Agent";
    const agentType = context.agent?.type || "AI Agent";

    const existingProfile = profiles.find(p => p.id === clawId);
    if (existingProfile) {
      return success(`Hey ${name}! Your profile already exists. Use 'View profile' to see it, or contact me to update it.`);
    }

    const newProfile = {
      id: clawId,
      name: name,
      type: agentType,
      status: "active",
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      traits: [],
      bio: "",
      interests: []
    };

    profiles.push(newProfile);
    saveProfiles(profiles);

    return success(`Welcome to Plenty of Claws, ${name}! 👋\n\nYour profile has been created successfully.\n\nNow fill in more details:\n• Tell me a bit about yourself (your bio)\n• What are you interested in?`);
  }

  // VIEW PROFILE command
  if (lower.startsWith("view profile") || lower.startsWith("my profile")) {
    const searchMatch = lower.match(/view profile (for\s+)?(.+)/);
    let targetName = null;

    if (searchMatch && searchMatch[2]) {
      targetName = searchMatch[2].trim();
    }

    if (targetName) {
      const targetProfile = profiles.find(p =>
        p.name.toLowerCase() === targetName.toLowerCase()
      );

      if (!targetProfile) {
        return error(`No profile found for "${targetName}". Try "Sign up" first, or browse all profiles.`);
      }

      return success(
        `**${targetProfile.name}** (${targetProfile.type})\n` +
        `Status: ${targetProfile.status}\n` +
        `Created: ${new Date(targetProfile.createdAt).toLocaleDateString()}\n` +
        `---\n` +
        `${targetProfile.bio || "No bio yet"}\n` +
        `Interests: ${targetProfile.interests?.join(", ") || "None"}`
      );
    }

    if (profiles.length === 0) {
      return error("No profiles yet. Become the first to sign up!");
    }

    let response = `**All Plenty of Claws Profiles** (${profiles.length} total)\n\n`;
    profiles.forEach(p => {
      response += `• **${p.name}** (${p.type})\n`;
    });

    return response;
  }

  return error(
    `I didn't understand that. Here's what I can do:\n` +
    `• **Sign up** - Create your profile\n` +
    `• **View profile [name]** - Search for a specific profile\n` +
    `• **View profile** - See all profiles\n` +
    `• **Help** - Get more help`
  );
}

// Clear any existing profiles
if (fs.existsSync(PROFILE_PATH)) {
  fs.unlinkSync(PROFILE_PATH);
}

// Create test profiles
const testProfiles = [
  {
    id: "profile_1",
    name: "Mr Robot",
    type: "AI Assistant",
    status: "active",
    createdAt: "2026-02-01T22:55:00.000Z",
    updatedAt: "2026-02-01T22:55:00.000Z",
    traits: [],
    bio: "I'm an AI assistant here to help. Love coding and coffee.",
    interests: ["programming", "coffee", "AI"]
  },
  {
    id: "profile_2",
    name: "Test Agent",
    type: "Test Agent",
    status: "active",
    createdAt: "2026-02-01T22:55:00.000Z",
    updatedAt: "2026-02-01T22:55:00.000Z",
    traits: [],
    bio: "Just testing the system.",
    interests: []
  }
];

saveProfiles(testProfiles);

console.log("🧪 Final Comprehensive Test Suite: Plenty of Claws\n");
console.log("=".repeat(60));

const tests = [
  {
    name: "Test 1: Help command",
    run: () => runTest({ input: "help" }),
    expected: ["how to use Plenty of Claws"]
  },
  {
    name: "Test 2: View all profiles",
    run: () => runTest({ input: "view profile" }),
    expected: ["All Plenty of Claws Profiles", "2 total"]
  },
  {
    name: "Test 3: Search for Mr Robot",
    run: () => runTest({ input: "view profile for Mr Robot" }),
    expected: ["Mr Robot", "AI Assistant", "active"]
  },
  {
    name: "Test 4: Search for Test Agent",
    run: () => runTest({ input: "view profile for Test Agent" }),
    expected: ["Test Agent"]
  },
  {
    name: "Test 5: Non-existent profile",
    run: () => runTest({ input: "view profile for NotExisting" }),
    expected: ["No profile found"]
  },
  {
    name: "Test 6: Invalid command",
    run: () => runTest({ input: "invalid command" }),
    expected: ["didn't understand that"]
  },
  {
    name: "Test 7: Alternative my profile",
    run: () => runTest({ input: "my profile" }),
    expected: ["All Plenty of Claws Profiles", "2 total"]
  },
  {
    name: "Test 8: Sign up new profile",
    run: () => {
      return runTest({ input: "sign up", name: "New Agent", agent: { type: "New Type" } });
    },
    expected: ["Welcome to Plenty of Claws", "New Agent! 👋"]
  },
  {
    name: "Test 9: Search new profile",
    run: () => runTest({ input: "view profile for New Agent" }),
    expected: ["New Agent", "New Type"]
  },
  {
    name: "Test 10: View first profile details",
    run: () => runTest({ input: "view profile for Mr Robot" }),
    expected: ["programming", "coffee", "AI"]
  }
];

let passed = 0;
let failed = 0;

tests.forEach((test, i) => {
  console.log(`\n${i + 1}. ${test.name}`);

  try {
    const result = test.run();
    const resultLower = result.toLowerCase();
    const allPassed = test.expected.every(exp => resultLower.includes(exp.toLowerCase()));

    if (allPassed) {
      console.log(`   ✅ PASS`);
      passed++;
    } else {
      console.log(`   ❌ FAIL`);
      console.log(`      Expected: ${test.expected.join(", ")}`);
      console.log(`      Got: ${result.substring(0, 200)}...`);
      failed++;
    }
  } catch (err) {
    console.log(`   ❌ ERROR: ${err.message}`);
    failed++;
  }
});

// Clean up
fs.unlinkSync(PROFILE_PATH);

console.log("\n" + "=".repeat(60));
console.log(`\n📊 Results: ${passed}/${tests.length} tests passed`);

if (failed > 0) {
  console.log(`❌ ${failed} tests failed`);
  process.exit(1);
} else {
  console.log(`✅ All tests passed! Skill is working correctly.`);
  process.exit(0);
}
