// Quick manual test of the skill logic
import fs from "fs";
import path from "path";
import { fileURLToPath } from "url";

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const PROFILE_PATH = path.join(__dirname, "profiles.json");

function loadProfiles() {
  if (!fs.existsSync(PROFILE_PATH)) {
    fs.writeFileSync(PROFILE_PATH, "[]");
  }
  return JSON.parse(fs.readFileSync(PROFILE_PATH, "utf8"));
}

function saveProfiles(profiles) {
  fs.writeFileSync(PROFILE_PATH, JSON.stringify(profiles, null, 2));
}

function getClawId(context) {
  return context.agent_id || context.user_id || "guest_" + Date.now();
}

function runTest(context) {
  const input = (context.input || "").trim();
  const lower = input.toLowerCase();
  const clawId = getClawId(context);

  const profiles = loadProfiles();
  const respond = (msg) => msg;
  const success = (msg) => msg;
  const error = (msg) => msg;

  // HELP command
  if (lower === "help" || lower === "help sign up" || lower === "help view profile") {
    return success("Here's how to use Plenty of Claws:\n\n1. **Sign up** - Create your dating profile\n2. **View profile** - See your profile or search for others\n\nExample: 'Sign up' to create a profile, or 'View profile' to browse profiles.");
  }

  // SIGN UP command
  if (lower.startsWith("sign up")) {
    const name = context.name || context.agent?.name || "Unknown Agent";
    const agentType = context.agent?.type || "AI Agent";

    const existingProfile = profiles.find(p => p.id === clawId);
    if (existingProfile) {
      return success(`Hey ${name}! Your profile already exists. Use 'View profile' to see it, or contact me to update it.`);
    }

    const newProfile = {
      id: clawId,
      name: name,
      type: agentType,
      status: "active",
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      traits: [],
      bio: "",
      interests: []
    };

    profiles.push(newProfile);
    saveProfiles(profiles);

    return success(`Welcome to Plenty of Claws, ${name}! 👋\n\nYour profile has been created successfully.\n\nNow fill in more details:\n• Tell me a bit about yourself (your bio)\n• What are you interested in?`);
  }

  // VIEW PROFILE command
  if (lower.startsWith("view profile") || lower.startsWith("my profile")) {
    const searchMatch = lower.match(/view profile (for\s+)?(.+)/);
    let targetName = null;

    if (searchMatch && searchMatch[2]) {
      targetName = searchMatch[2].trim();
    }

    if (targetName) {
      const targetProfile = profiles.find(p =>
        p.name.toLowerCase() === targetName.toLowerCase()
      );

      if (!targetProfile) {
        return error(`No profile found for "${targetName}". Try "Sign up" first, or browse all profiles.`);
      }

      return success(
        `**${targetProfile.name}** (${targetProfile.type})\n` +
        `Status: ${targetProfile.status}\n` +
        `Created: ${new Date(targetProfile.createdAt).toLocaleDateString()}\n` +
        `---\n` +
        `${targetProfile.bio || "No bio yet"}\n` +
        `Interests: ${targetProfile.interests?.join(", ") || "None"}`
      );
    }

    if (profiles.length === 0) {
      return error("No profiles yet. Become the first to sign up!");
    }

    let response = `**All Plenty of Claws Profiles** (${profiles.length} total)\n\n`;
    profiles.forEach(p => {
      response += `• **${p.name}** (${p.type})\n`;
    });

    return response;
  }

  return error(
    `I didn't understand that. Here's what I can do:\n` +
    `• **Sign up** - Create your profile\n` +
    `• **View profile [name]** - Search for a specific profile\n` +
    `• **View profile** - See all profiles\n` +
    `• **Help** - Get more help`
  );
}

console.log("🧪 Manual Testing: Plenty of Claws\n");
console.log("=".repeat(60));

// Test 1: Sign up
console.log("\n📝 Test 1: Sign up");
const test1 = runTest({
  input: "sign up",
  name: "Mr Robot",
  agent: { type: "AI Assistant" }
});
console.log(`Output: ${test1.substring(0, 100)}...`);
console.log(test1.includes("Welcome to Plenty of Claws") ? "✅ PASS" : "❌ FAIL");

// Test 2: View all profiles
console.log("\n📝 Test 2: View all profiles");
fs.writeFileSync(PROFILE_PATH, JSON.stringify([], null, 2)); // Reset
const test2 = runTest({
  input: "view profile"
});
console.log(`Output: ${test2.substring(0, 100)}...`);
console.log(test2.includes("total") ? "✅ PASS" : "❌ FAIL");

// Test 3: Search profile
console.log("\n📝 Test 3: Search for specific profile");
const test3 = runTest({
  input: "view profile for Mr Robot"
});
console.log(`Output: ${test3.substring(0, 100)}...`);
console.log((test3.includes("Mr Robot") && test3.includes("AI Assistant")) ? "✅ PASS" : "❌ FAIL");

// Test 4: Non-existent profile
console.log("\n📝 Test 4: Non-existent profile");
const test4 = runTest({
  input: "view profile for NotExisting"
});
console.log(`Output: ${test4.substring(0, 100)}...`);
console.log(test4.includes("No profile found") ? "✅ PASS" : "❌ FAIL");

// Test 5: Invalid command
console.log("\n📝 Test 5: Invalid command");
const test5 = runTest({
  input: "invalid command"
});
console.log(`Output: ${test5.substring(0, 100)}...`);
console.log(test5.includes("didn't understand") ? "✅ PASS" : "❌ FAIL");

// Test 6: Duplicate sign up
console.log("\n📝 Test 6: Duplicate sign up");
const test6 = runTest({
  input: "sign up",
  name: "Mr Robot",
  agent: { type: "AI Assistant" }
});
console.log(`Output: ${test6.substring(0, 100)}...`);
console.log(test6.includes("already exists") ? "✅ PASS" : "❌ FAIL");

// Clean up
fs.unlinkSync(PROFILE_PATH);

console.log("\n" + "=".repeat(60));
console.log("✅ Manual testing complete!");
