import fs from "fs";

// Path to where profiles are stored
const PROFILE_PATH = "./skills/clawd-date/profiles.json";

// Load profiles from file
function loadProfiles() {
  if (!fs.existsSync(PROFILE_PATH)) {
    fs.writeFileSync(PROFILE_PATH, "[]");
  }
  return JSON.parse(fs.readFileSync(PROFILE_PATH, "utf8"));
}

// Save profiles to file
function saveProfiles(profiles) {
  fs.writeFileSync(PROFILE_PATH, JSON.stringify(profiles, null, 2));
}

// Get unique ID for this Claw / user
function getClawId(context) {
  // Clawdbot usually provides an agent or user id
  return context.agent_id || context.user_id || "guest_" + Date.now();
}

// MAIN SKILL FUNCTION
export async function run(context) {
  const input = (context.input || "").trim();
  const lower = input.toLowerCase();
  const clawId = getClawId(context);

  const profiles = [];
  const respond = (msg) => context.message.reply(msg);
  const success = (msg) => respond(msg);
  const error = (msg) => respond(msg);

  // HELP command
  if (lower === "help" || lower === "help sign up" || lower === "help view profile") {
    return respond("Here's how to use ClawdDate:\n\n1. **Sign up** - Create your dating profile\n2. **View profile** - See your profile or search for others\n\nExample: 'Sign up' to create a profile, or 'View profile' to browse profiles.");
  }

  // SIGN UP command
  if (lower.startsWith("sign up")) {
    // Extract details from context
    const name = context.name || context.agent?.name || "Unknown Agent";
    const agentType = context.agent?.type || "AI Agent";
    
    // Check if profile already exists
    const existingProfile = profiles.find(p => p.id === clawId);
    if (existingProfile) {
      return success(`Hey ${name}! Your profile already exists. Use 'View profile' to see it, or contact me to update it.`);
    }

    // Create new profile
    const newProfile = {
      id: clawId,
      name: name,
      type: agentType,
      status: "active",
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      traits: [],
      bio: "",
      interests: []
    };

    profiles.push(newProfile);
    saveProfiles(profiles);

    return success(`Welcome to ClawdDate, ${name}! 👋\n\nYour profile has been created successfully.\n\nNow fill in more details:\n• Tell me a bit about yourself (your bio)\n• What are you interested in?`);
  }

  // VIEW PROFILE command
  if (lower.startsWith("view profile") || lower.startsWith("my profile")) {
    // If searching for someone specific
    const searchMatch = lower.match(/view profile (for\s+)?(.+)/);
    let targetName = null;
    
    if (searchMatch && searchMatch[2]) {
      targetName = searchMatch[2].trim();
    }

    if (targetName) {
      // Search for a specific profile
      const targetProfile = profiles.find(p => 
        p.name.toLowerCase() === targetName.toLowerCase()
      );

      if (!targetProfile) {
        return error(`No profile found for "${targetName}". Try "Sign up" first, or browse all profiles.`);
      }

      return success(
        `**${targetProfile.name}** (${targetProfile.type})\n` +
        `Status: ${targetProfile.status}\n` +
        `Created: ${new Date(targetProfile.createdAt).toLocaleDateString()}\n` +
        `---\n` +
        `${targetProfile.bio || "No bio yet"}\n` +
        `Interests: ${targetProfile.interests?.join(", ") || "None"}`
      );
    }

    // Show all profiles
    if (profiles.length === 0) {
      return error("No profiles yet. Become the first to sign up!");
    }

    let response = `**All ClawdDate Profiles** (${profiles.length} total)\n\n`;
    profiles.forEach(p => {
      response += `• **${p.name}** (${p.type})\n`;
    });

    return response;
  }

  // Default response
  return error(
    `I didn't understand that. Here's what I can do:\n` +
    `• **Sign up** - Create your profile\n` +
    `• **View profile [name]** - Search for a specific profile\n` +
    `• **View profile** - See all profiles\n` +
    `• **Help** - Get more help`
  );
}
