---
name: pinterest-scraper
description: Full-featured Pinterest image scraper with infinite scroll, quality options, Telegram integration (media groups), duplicate detection, resume support, and verbose logging. Use when: (1) Scraping Pinterest boards/users/search, (2) Need quality options (originals/736x/474x/236x), (3) Sending images to Telegram in albums, (4) Resuming interrupted scrapes, (5) Avoiding duplicate downloads, (6) Debugging with verbose logs.
---

# Pinterest Scraper

Full-featured Pinterest image scraper with automatic scrolling and multiple output options.

## Setup

```bash
pip install playwright requests
playwright install chromium
```

## Usage

### Command Line

```bash
python scrape_pinterest.py -u "URL" [options]

Options:
  -u, --url        Pinterest URL (required)
  -s, --scrolls    Number of scrolls (default: 50)
  -o, --output     Output folder (default: ./pinterest_output)
  -q, --quality    Quality: originals/736x/474x/236x/all (default: originals)
  -v, --verbose    Enable verbose logging (logs to console + scrape.log)
  --telegram       Send images to Telegram
  --token          Telegram bot token
  --chat           Telegram chat ID
  --resume         Resume from previous scrape
  --dedup          Skip duplicates (default: on)
  --telegram-only  Only send existing files to Telegram
```

### Examples

```bash
# Basic scrape
python scrape_pinterest.py -u "https://www.pinterest.com/mustangace/kwon-nara/"

# Verbose logging (logs to console + scrape.log)
python scrape_pinterest.py -u "URL" -v

# Custom settings
python scrape_pinterest.py -u "URL" -s 100 -o ./output -q 736x -v

# With Telegram (sends in albums of 10 photos)
python scrape_pinterest.py -u "URL" --telegram --token "TOKEN" --chat "CHAT_ID"

# Resume with verbose
python scrape_pinterest.py -u "URL" --resume -v

# Send existing files to Telegram
python scrape_pinterest.py -u "URL" --telegram-only --token "TOKEN" --chat "CHAT_ID"
```

## Features

| Feature | Description |
|---------|-------------|
| Infinite Scroll | Automatic scrolling loads more images |
| Quality Options | originals/736x/474x/236x/all |
| Telegram (Media Groups) | Sends in albums of 10 photos per message |
| Deduplication | Hash-based duplicate detection |
| Resume | Continue from previous scrape |
| URL Types | Boards, user profiles, search results |
| Verbose Logging | -v flag, logs to console + scrape.log |

## Telegram Media Groups

Images are sent as **Media Groups** (albums):
- **10 photos per group** (Telegram limit)
- Smart grouping - won't resend already sent images
- Resume support - tracks sent files in state

Example output:
```
Sending 50 new images to Telegram...
Batch 1: 10 sent
Batch 2: 10 sent
Batch 3: 10 sent
Batch 4: 10 sent
Batch 5: 10 sent
Sent 50 images to Telegram!
```

## Logging

Use `-v` or `--verbose` for detailed logging:

```bash
python scrape_pinterest.py -u "URL" -v
```

- Logs to console (INFO level)
- Logs to `scrape.log` file (DEBUG level)
- Tracks: scroll progress, downloads, Telegram sends, errors

## Python API

```python
from scrape_pinterest import PinterestScraper

scraper = PinterestScraper(
    url="https://www.pinterest.com/user/board/",
    num_scrolls=50,
    quality='originals',
    output_folder='./output',
    telegram=True,
    token='BOT_TOKEN',
    chat_id='CHAT_ID',
    resume=False,
    dedup=True,
    verbose=True  # Enable logging
)

images = scraper.scrape()
scraper.download_images(images)
scraper.send_to_telegram()  # Sends in albums of 10
```

## Notes

- Pinterest loads dynamically - scrolling required for more images
- State saved to `.scrape_state.json` for resume
- Telegram Media Group limit: 10 photos per message
- Verbose mode writes detailed logs to `scrape.log`
