# Financial Goal Management

This document defines how PhoenixClaw Ledger manages financial goals, tracks progress, and integrates with the daily journal and annual reports.

## Goal Types

PhoenixClaw Ledger supports four primary goal types:

### 1. Savings Goal (`savings`)
Long-term accumulation of funds for a specific purpose.
- **Example:** "存 5 万旅游基金" (Save 50,000 for travel fund)
- **Key Metrics:** Target amount, current balance, deadline.

### 2. Budget Control Goal (`budget_control`)
Strict limits on specific categories or behaviors over a period.
- **Example:** "本月餐饮不超 2000" (Food spending under 2,000 this month)
- **Key Metrics:** Category/Tag, spending limit, time period (weekly/monthly).

### 3. Habit Goal (`habit`)
Behavioral goals focused on consistency rather than currency.
- **Example:** "连续 7 天不叫外卖" (No takeout for 7 consecutive days)
- **Key Metrics:** Target behavior, current streak, target streak.

### 4. Wishlist Goal (`wishlist`)
Short-to-medium term savings for a specific item.
- **Example:** "存够买 MacBook" (Save enough for MacBook)
- **Key Metrics:** Item cost, allocated savings, progress percentage.

## Storage Schema

Goals are stored in `~/PhoenixClaw/Finance/goals.yaml`.

```yaml
# goals.yaml
goals:
  - id: "goal_001"
    name: "Travel Fund 2026"
    type: savings
    status: active            # active | completed | cancelled | paused
    currency: CNY
    target_amount: 50000
    current_amount: 12500
    start_date: "2026-01-01"
    deadline: "2026-12-31"
    tags: [travel, savings]
    created_at: "2025-12-20T10:00:00Z"

  - id: "goal_002"
    name: "Dine-out Limit"
    type: budget_control
    status: active
    category: food
    limit_amount: 2000
    period: monthly
    current_spent: 850
    period_start: "2026-02-01"
    period_end: "2026-02-28"

  - id: "goal_003"
    name: "Anti-Takeout Habit"
    type: habit
    status: active
    behavior: "no_takeout"
    target_streak: 7
    current_streak: 4
    best_streak: 12
    last_triggered: "2026-02-02"

  - id: "goal_004"
    name: "MacBook Pro"
    type: wishlist
    status: active
    item_cost: 15000
    allocated_amount: 3000
    priority: medium         # low | medium | high
```

## Goal Workflow (CRUD)

### Create
- **Manual:** User adds goal via chat or direct YAML edit.
- **Automatic:** Ledger detects goal intent from conversation (e.g., "I want to save ¥10k for a bike").
- **ID Generation:** `goal_` prefix followed by incrementing index or hash.

### Read
- **Daily Scan:** PhoenixClaw Core scans `goals.yaml` during the 10 PM run.
- **Query:** User asks "How are my goals doing?".

### Update
- **Transaction-Driven:** Savings/Wishlist goals update when transactions with specific tags or descriptions are detected.
- **Budget-Driven:** `budget_control` goals update in real-time as transactions are categorized.
- **Behavioral:** `habit` goals update when daily memory analysis confirms the behavior (or lack thereof).

### Delete/Complete
- **Auto-Complete:** Triggered when `progress >= 100%`.
- **Manual Deletion:** User explicitly requests to stop tracking a goal.

## Progress Calculation

### Savings & Wishlist
Progress is a simple percentage of target reached.
```
progress_percent = (current_amount / target_amount) * 100
```
- **Example:** ¥12,500 / ¥50,000 = 25%

### Budget Control
Progress measures how much of the "allowance" has been consumed.
```
consumption_percent = (current_spent / limit_amount) * 100
status = "on_track" if (consumption_percent <= time_elapsed_percent) else "warning"
```
- **Time Elapsed %:** `(days_since_start / total_days_in_period) * 100`

### Habit
Progress is measured against the target streak.
```
streak_percent = (current_streak / target_streak) * 100
```

## Milestone Notifications

Milestones are highlighted in the daily journal's "Financial Growth" section.

| Milestone | Condition | Notification Style |
|-----------|-----------|--------------------|
| **Quarter-Way** | Progress reaches 25% | `[!insight]` Encouraging note |
| **Half-Way** | Progress reaches 50% | `[!success]` Celebration note |
| **Home Stretch** | Progress reaches 75% | `[!milestone]` Motivation boost |
| **Completed** | Progress reaches 100% | `[!success]` Achievement block |

### Notification Rules
- Only trigger once per milestone per goal.
- Max 2 milestone notifications in a single daily journal.
- Priority: Completed > 75% > 50% > 25%.

## Edge Case Handling

### 1. Deadline Passed
If `current_date > deadline` and `progress < 100%`:
- Status changes to `paused` (requires user intervention).
- Journal surfaces a `[!reflection]` callout asking if the goal is still relevant or needs a deadline extension.

### 2. Over-Budget (Budget Control)
If `current_spent > limit_amount`:
- Status flagged as `exceeded`.
- Immediate `[!alert]` in the daily journal.
- Calculation of required reduction in other categories to compensate.

### 3. Habit Break
If a habit behavior is violated:
- `current_streak` resets to 0.
- `[!reflection]` callout to identify the trigger/reason for the break.

### 4. Goal Modification
If a user changes a goal's target mid-way:
- Archive the previous progress state.
- Recalculate percentages from the new baseline.

## Integration Patterns

### Journal Output
Goals contribute to the `{{FINANCE_SUMMARY}}` and `{{GROWTH_NOTES}}` blocks:
- **Daily:** Milestone alerts and habit streak updates.
- **Weekly:** Summary of progress across all active goals.
- **Yearly:** Final status of all goals tracked during the year.

### Annual Report Format
Integration with `assets/yearly-report.md`:
```markdown
## Financial Goals

- Travel Fund 2026: ¥12,500/¥50,000 (25%) 🧗
- Dine-out Limit: ¥850/¥2,000 (42%) ✅
- Anti-Takeout Habit: 4/7 days streak (57%) 🔥
- MacBook Pro: ¥3,000/¥15,000 (20%) 🔋
```
