# Budget Tracking

This document defines how PhoenixClaw Ledger manages budgets, tracks progress, and generates alerts.

## Budget Configuration

### User Setup

Budgets are configured in `~/.phoenixclaw/config.yaml`:

```yaml
plugins:
  phoenixclaw-ledger:
    budget:
      enabled: true
      period: monthly           # monthly | weekly | custom
      currency: CNY
      total: 5000               # Total monthly budget
      
      # Category allocations (optional)
      categories:
        food: 1500
        transport: 500
        shopping: 800
        entertainment: 400
        # Unspecified categories share remaining budget
      
      # Alert thresholds
      alerts:
        warn_at: 80             # Warn at 80% spent
        critical_at: 95         # Critical at 95% spent
        daily_limit: 300        # Optional daily spending limit
```

### Storage Location

```
~/PhoenixClaw/Finance/
└── budget.yaml                 # Budget configuration and state
```

### Budget State Schema

```yaml
# budget.yaml
config:
  period: monthly
  total: 5000
  currency: CNY
  categories: {...}
  
state:
  current_period: "2026-02"
  period_start: "2026-02-01"
  period_end: "2026-02-28"
  
  spent:
    total: 3280
    by_category:
      food: 1200
      transport: 380
      shopping: 850
      entertainment: 250
      other: 600
      
  remaining:
    total: 1720
    days_left: 12
    daily_allowance: 143.33
```

## Progress Calculation

### Basic Metrics

```yaml
budget_metrics:
  # Absolute values
  total_budget: 5000
  total_spent: 3280
  total_remaining: 1720
  
  # Percentages
  percent_spent: 65.6
  percent_remaining: 34.4
  
  # Time-based
  days_elapsed: 16
  days_total: 28
  percent_time_elapsed: 57.1
  
  # Pace analysis
  expected_spend_by_now: 2857    # Budget * (days_elapsed / days_total)
  actual_vs_expected: -423       # Negative = under budget
  pace_status: on_track          # under | on_track | over
```

### Pace Status Calculation

```
pace_ratio = (percent_spent / percent_time_elapsed)

if pace_ratio < 0.9:
    status = "under"      # Spending slower than time
elif pace_ratio > 1.1:
    status = "over"       # Spending faster than time  
else:
    status = "on_track"   # Within 10% of expected
```

## Alert System

### Alert Levels

| Level | Trigger | Icon | Action |
|-------|---------|------|--------|
| **Info** | Daily summary | ℹ️ | Show in journal |
| **Warning** | 80% budget used | ⚠️ | Highlight in journal |
| **Critical** | 95% budget used | 🚨 | Prominent warning |
| **Exceeded** | Over 100% | ❌ | Strong warning |

### Alert Messages

```yaml
alerts:
  info:
    - "Today's spending: ¥{daily_total}"
    - "This week: ¥{weekly_total}"
    
  warning:
    threshold: 80
    messages:
      - "Budget alert: {percent}% spent with {days} days remaining"
      - "Consider reducing discretionary spending"
      
  critical:
    threshold: 95
    messages:
      - "⚠️ Budget nearly exhausted: {percent}% spent"
      - "Only ¥{remaining} left for {days} days"
      
  exceeded:
    messages:
      - "❌ Budget exceeded by ¥{overage}"
      - "Total spending: ¥{total} / ¥{budget}"
```

### Category-Specific Alerts

```yaml
category_alerts:
  - category: food
    threshold: 90
    message: "Food budget at {percent}%"
    
  - category: shopping
    threshold: 75
    message: "Shopping nearing limit"
```

## Daily Allowance

### Calculation

```
daily_allowance = remaining_budget / days_remaining

# With buffer for unexpected expenses
safe_daily = daily_allowance * 0.9
```

### Display Format

```markdown
**Daily Budget Guide**
- Remaining: ¥1,720
- Days left: 12
- Daily allowance: ¥143/day
- Safe spending: ¥129/day
```

## Period Management

### Monthly Reset

```yaml
monthly_reset:
  trigger: first_day_of_month
  actions:
    - archive_previous_period
    - reset_spent_totals
    - recalculate_allowances
    - generate_monthly_report
```

### Rollover Options

```yaml
rollover:
  enabled: false          # Whether unused budget rolls over
  max_rollover: 500       # Maximum rollover amount
  rollover_to: savings    # savings | next_month | category
```

## Budget Reports

### Daily Summary (in Journal)

```markdown
## 💰 Financial Summary

**Today:** ¥449 spent
**This Month:** ¥3,280 / ¥5,000 (66%)

> [!budget] 📊 Budget Status
> - Pace: On track ✓
> - Daily allowance: ¥143
> - Remaining: ¥1,720 for 12 days
```

### Weekly Summary

```markdown
## 📊 Weekly Budget Report

| Category | Spent | Budget | Status |
|----------|-------|--------|--------|
| Food | ¥680 | ¥350/week | ⚠️ Over |
| Transport | ¥120 | ¥125/week | ✅ OK |
| Shopping | ¥450 | ¥200/week | ⚠️ Over |

**Weekly Total:** ¥1,250 / ¥1,250 (100%)
```

### Monthly Report

See `assets/monthly-report.md` for full template.

## Smart Insights

### Pattern-Based Recommendations

```yaml
insights:
  - pattern: "weekend_spike"
    condition: "weekend_spending > weekday_spending * 2"
    message: "Weekend spending is significantly higher than weekdays"
    suggestion: "Consider setting a separate weekend budget"
    
  - pattern: "early_exhaustion"
    condition: "percent_spent > 80 AND percent_time < 60"
    message: "Budget depleting faster than expected"
    suggestion: "Try to reduce spending in remaining days"
    
  - pattern: "category_imbalance"
    condition: "single_category > 50% of total"
    message: "{category} dominates your spending"
    suggestion: "Review if this aligns with your priorities"
```

### Projections

```yaml
projections:
  # End-of-month projection based on current pace
  projected_total: 5125
  projected_overage: 125
  
  # What-if scenarios
  if_reduce_daily_by: 50
  then_end_month_at: 4525
  
  # Historical comparison
  vs_last_month: +8%
  vs_3_month_avg: +12%
```

## Integration with Journal

### Section Output

Ledger exports budget info to PhoenixClaw Core:

```yaml
plugin_output:
  section_id: finance
  section_title: "💰 Financial Summary"
  section_order: 45
  
  budget_summary:
    spent_today: 449
    spent_month: 3280
    budget_total: 5000
    percent_used: 65.6
    status: on_track
    alerts: []
    
  content: |
    (Rendered markdown content)
```

### Conditional Display

```yaml
display_rules:
  # Always show if there's spending
  show_if: daily_spending > 0
  
  # Expand details if notable
  expand_if:
    - daily_spending > daily_allowance
    - budget_percent > 80
    - any_alerts_active
```

---
