#!/usr/bin/env node
/**
 * Persona Evolution Skill - Weekly Evolution Report
 * 
 * Generates weekly summary and suggests SOUL.md updates
 */

import { readFileSync, writeFileSync, existsSync, readdirSync } from 'fs';
import { join } from 'path';

const WORKSPACE = process.cwd();
const PERSONA_DIR = join(WORKSPACE, 'PERSONA');
const EVOLVES_DIR = join(PERSONA_DIR, 'evolves');
const SOUL_PATH = join(WORKSPACE, 'SOUL.md');

function loadJson(filename) {
  try {
    return JSON.parse(readFileSync(join(EVOLVES_DIR, filename), 'utf-8'));
  } catch {
    return null;
  }
}

function formatDate(date) {
  return date.toISOString().split('T')[0];
}

function main() {
  console.log('📊 Generating Weekly Evolution Report...\n');
  
  const patterns = loadJson('patterns.json');
  if (!patterns) {
    console.log('⚠️  No pattern data found. Run analyze-session first.');
    return;
  }
  
  // Get date range
  const now = new Date();
  const weekAgo = new Date(now - 7 * 24 * 60 * 60 * 1000);
  
  // Count sessions
  const sessionFiles = readdirSync(EVOLVES_DIR)
    .filter(f => f.startsWith('session-'));
  
  const report = `# Evolution Report: ${formatDate(weekAgo)} to ${formatDate(now)}

## 📈 Engagement Metrics

**Total Sessions:** ${patterns.session_count}
**This Week:** ${sessionFiles.length}

### Communication Style
- **Preferred Format:** ${patterns.communication_style.preferred_format}
- **Detail Level:** ${patterns.communication_style.detail_level}

### Trust Level: ${patterns.trust_indicators.trust_level}
- Autonomous actions: ${patterns.trust_indicators.autonomous_actions}
- Corrections made: ${patterns.trust_indicators.corrections_made}

## 🎭 Emotional Context

**Dominant Mood:** ${patterns.emotional_context.dominant_mood}

**Recent Stress Signals:**
${patterns.emotional_context.stress_signals.slice(-3).map(s => `- ${s.date}: ${s.markers.join(', ')}`).join('\n') || 'None detected'}

**Celebration Moments:**
${patterns.emotional_context.celebration_moments.slice(-3).map(m => `- ${m.date}: ${m.markers.join(', ')}`).join('\n') || 'None recorded'}

## 💡 Suggested Updates

### To SOUL.md:

\`\`\`diff
+ ## Communication Preferences (Detected)
+ - Prefers ${patterns.communication_style.preferred_format} format
+ - Detail level: ${patterns.communication_style.detail_level}
+ 
+ ## Current Priorities
${patterns.engagement_signals.enthusiasm_topics.slice(-5).map(t => `+ - ${t}`).join('\n')}
\`\`\`

### To PERSONA/interests.md:

Update priority rankings based on engagement frequency.

## 🎯 Recommendations

${patterns.trust_indicators.autonomous_actions > 10 
  ? '✅ **Trust is established** — You can safely increase autonomous actions'
  : '📊 **Trust building** — Continue asking permission for major actions'}

${patterns.emotional_context.stress_signals.length > 2
  ? '⚠️ **Stress detected** — Consider proactive check-ins during busy periods'
  : '✅ **Low stress** — Good time for creative/experimental work'}

## 🔄 Next Steps

1. Review suggested SOUL.md updates above
2. Approve or modify as needed
3. Continue normal interactions — I'll keep learning!

---
*Generated by Persona Evolution Skill*
`;

  const reportPath = join(EVOLVES_DIR, 'weekly-report.md');
  writeFileSync(reportPath, report);
  
  console.log(report);
  console.log(`\n📝 Report saved to: ${reportPath}`);
}

main();
