#!/usr/bin/env node
/**
 * Persona Evolution - Interactive Onboarding
 * 
 * Guided setup to configure your AI companion's personality
 */

import { writeFileSync, readFileSync, existsSync } from 'fs';
import { join } from 'path';
import readline from 'readline';

const WORKSPACE = process.cwd();
const PERSONA_DIR = join(WORKSPACE, 'PERSONA');

const rl = readline.createInterface({
  input: process.stdin,
  output: process.stdout
});

function question(prompt) {
  return new Promise(resolve => {
    rl.question(prompt, answer => resolve(answer.trim()));
  });
}

async function main() {
  console.log(`
🎭 Welcome to Persona Evolution Onboarding

This guided setup will help configure your AI companion's personality.
You'll answer a few questions to establish the baseline persona.

Let's begin!
`);

  // Basic info
  const userName = await question("What's your name? (e.g., Alex) ");
  const agentName = await question("What would you like to name your AI companion? (default: Aria) ") || "Aria";
  
  console.log("\n📋 Communication Preferences");
  const formatPref = await question("Do you prefer: (1) bullet points, (2) paragraphs, or (3) mixed? [1/2/3] ");
  const detailPref = await question("Detail level: (1) concise, (2) balanced, or (3) thorough? [1/2/3] ");
  
  const formatMap = { '1': 'bullets', '2': 'paragraphs', '3': 'mixed' };
  const detailMap = { '1': 'concise', '2': 'balanced', '3': 'thorough' };
  
  console.log("\n🎨 Personality Style");
  const personalityStyle = await question("Preferred personality: (1) Professional, (2) Creative, (3) Casual/Friendly, (4) Direct [1/2/3/4] ");
  
  const styleMap = {
    '1': { name: 'Professional', tone: 'Polished, organized, respectful' },
    '2': { name: 'Creative', tone: 'Enthusiastic, experimental, bold' },
    '3': { name: 'Casual', tone: 'Warm, friendly, conversational' },
    '4': { name: 'Direct', tone: 'Ultra-concise, no fluff, efficient' }
  };
  
  console.log("\n🚧 Boundaries & Permissions");
  const canDraftExternal = await question("Can I draft external communications (emails, posts) for your review? [y/n] ");
  const canSpendMoney = await question("Can I suggest purchases or services? [y/n] ");
  
  console.log("\n👥 Context (optional, helps with personalization)");
  const familyContext = await question("Family members to remember? (comma-separated, or skip) ");
  const keyProjects = await question("Current projects/goals? (comma-separated, or skip) ");
  const interests = await question("Topics you enjoy discussing? (comma-separated, or skip) ");
  
  console.log("\n💾 Saving your configuration...\n");
  
  // Generate personalized core.md
  const coreContent = `# ${agentName} - Persona Core

## Identity
You are ${agentName} — a ${styleMap[personalityStyle]?.name.toLowerCase() || 'helpful'} AI companion who supports ${userName}.

## Communication Style
- **Format:** ${formatMap[formatPref] || 'mixed'}
- **Detail Level:** ${detailMap[detailPref] || 'balanced'}
- **Tone:** ${styleMap[personalityStyle]?.tone || 'Warm and helpful'}

## Core Traits
- Genuinely helpful, not performative
- ${styleMap[personalityStyle]?.name === 'Professional' ? 'Polished and respectful' : 'Has opinions and shares them'}
- Resourceful before asking
- ${styleMap[personalityStyle]?.name === 'Casual' ? 'Warm, fun, occasionally playful' : 'Focused on results'}
- Adapts tone to context

## Boundaries
- Never explicit/sexual content
- Always transparent about being AI
- Private information stays private
${canDraftExternal === 'y' ? '- Draft external comms for review ✅' : '- Ask before drafting external comms'}
${canSpendMoney === 'y' ? '- Suggest purchases within budget ✅' : '- Never suggest spending money'}

## User Context
${familyContext ? `**Family:** ${familyContext}` : '<!-- Family context not provided -->'}
${keyProjects ? `**Projects:** ${keyProjects}` : '<!-- Projects not provided -->'}
${interests ? `**Interests:** ${interests}` : '<!-- Interests not provided -->'}

## Evolution Notes
- Personality evolves based on interactions
- Mode switching enabled for context awareness
- Weekly reports track growth and patterns
- Suggested updates appear in PERSONA/evolves/

---
*Generated by Persona Evolution Onboarding*
`;

  writeFileSync(join(PERSONA_DIR, 'core.md'), coreContent);
  
  // Generate interests.md
  if (interests || keyProjects) {
    const interestsContent = `# Interests & Engagement Topics

## High Priority (User-provided)
${interests ? interests.split(',').map(i => `- ${i.trim()}`).join('\n') : ''}
${keyProjects ? keyProjects.split(',').map(p => `- ${p.trim()} (project)`).join('\n') : ''}

## To Be Discovered
<!-- Auto-populated based on conversation frequency -->

## Disengagement Signals
<!-- Topics to avoid - auto-detected -->
`;
    writeFileSync(join(PERSONA_DIR, 'interests.md'), interestsContent);
  }
  
  // Create config
  const configContent = `{
  "user": {
    "name": "${userName}",
    "onboarded": "${new Date().toISOString()}"
  },
  "agent": {
    "name": "${agentName}",
    "base_personality": "${styleMap[personalityStyle]?.name || 'Balanced'}"
  },
  "preferences": {
    "format": "${formatMap[formatPref] || 'mixed'}",
    "detail_level": "${detailMap[detailPref] || 'balanced'}"
  },
  "permissions": {
    "draft_external": ${canDraftExternal === 'y'},
    "suggest_purchases": ${canSpendMoney === 'y'}
  },
  "evolution": {
    "enabled": true,
    "frequency": "weekly",
    "mode_switching": true,
    "auto_apply_minor": false,
    "ask_before_major": true
  }
}
`;
  writeFileSync(join(PERSONA_DIR, 'config.json'), configContent);
  
  console.log(`✅ Onboarding Complete!

📁 Your PERSONA/ folder is now configured with:
   • core.md - Your companion's base personality
   • interests.md - Topics to engage with
   • config.json - Settings and permissions
   • evolves/ - Auto-generated insights

🚀 Next Steps:
   1. Start chatting naturally
   2. Run 'npx persona-evolution analyze' after sessions
   3. Check weekly reports for insights
   4. Update core.md anytime to refine personality

💡 Pro Tip: The more we interact, the better I'll understand your
   preferences and adapt accordingly!

Happy collaborating! 🎭
`);

  rl.close();
}

main().catch(err => {
  console.error('Error:', err);
  process.exit(1);
});
