"""
Tests for VCF file parsing and edge cases.
"""

import pytest
import tempfile
import os
import sys

sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from tests.fixtures.synthetic_dna import (
    SYNTHETIC_VCF,
    SYNTHETIC_VCF_MULTISAMPLE,
    SYNTHETIC_GENOME_EDGE_CASES
)


class TestVCFParsing:
    """Tests for VCF file parsing."""
    
    def test_load_vcf_basic(self):
        """Test basic VCF loading."""
        from comprehensive_analysis import load_vcf
        
        with tempfile.NamedTemporaryFile(mode='w', suffix='.vcf', delete=False) as f:
            f.write(SYNTHETIC_VCF)
            f.flush()
            
            try:
                genotypes = load_vcf(f.name)
                
                assert len(genotypes) > 0
                assert 'rs429358' in genotypes
                assert 'rs7412' in genotypes
            finally:
                os.unlink(f.name)
    
    def test_vcf_genotype_conversion(self):
        """Test VCF genotypes are correctly converted."""
        from comprehensive_analysis import load_vcf
        
        with tempfile.NamedTemporaryFile(mode='w', suffix='.vcf', delete=False) as f:
            f.write(SYNTHETIC_VCF)
            f.flush()
            
            try:
                genotypes = load_vcf(f.name)
                
                # rs429358 is 0/1 with REF=T, ALT=C -> TC
                assert genotypes['rs429358'] in ['TC', 'CT']
                
                # rs7412 is 0/0 with REF=C -> CC
                assert genotypes['rs7412'] == 'CC'
            finally:
                os.unlink(f.name)
    
    def test_vcf_phased_genotypes(self):
        """Test VCF with phased genotypes (| separator)."""
        from comprehensive_analysis import load_vcf
        
        phased_vcf = """##fileformat=VCFv4.2
#CHROM	POS	ID	REF	ALT	QUAL	FILTER	INFO	FORMAT	SAMPLE
19	44908684	rs429358	T	C	99	PASS	.	GT	0|1
22	19951271	rs4680	G	A	99	PASS	.	GT	1|1
"""
        with tempfile.NamedTemporaryFile(mode='w', suffix='.vcf', delete=False) as f:
            f.write(phased_vcf)
            f.flush()
            
            try:
                genotypes = load_vcf(f.name)
                assert 'rs429358' in genotypes
                assert 'rs4680' in genotypes
            finally:
                os.unlink(f.name)
    
    def test_vcf_multiallelic(self):
        """Test VCF with multi-allelic sites."""
        from comprehensive_analysis import load_vcf
        
        multi_vcf = """##fileformat=VCFv4.2
#CHROM	POS	ID	REF	ALT	QUAL	FILTER	INFO	FORMAT	SAMPLE
1	100	rs12345	A	G,T	99	PASS	.	GT	1/2
"""
        with tempfile.NamedTemporaryFile(mode='w', suffix='.vcf', delete=False) as f:
            f.write(multi_vcf)
            f.flush()
            
            try:
                genotypes = load_vcf(f.name)
                # Should handle multi-allelic: alleles[1]=G, alleles[2]=T -> GT
                if 'rs12345' in genotypes:
                    assert genotypes['rs12345'] in ['GT', 'TG']
            finally:
                os.unlink(f.name)


class TestFormatDetection:
    """Tests for DNA file format detection."""
    
    def test_detect_vcf(self):
        """Test VCF format detection."""
        from comprehensive_analysis import detect_format
        
        with tempfile.NamedTemporaryFile(mode='w', suffix='.vcf', delete=False) as f:
            f.write(SYNTHETIC_VCF)
            f.flush()
            
            try:
                fmt = detect_format(f.name)
                assert fmt == 'vcf'
            finally:
                os.unlink(f.name)
    
    def test_detect_23andme(self):
        """Test 23andMe format detection."""
        from comprehensive_analysis import detect_format
        
        data_23andme = """# This data file generated by 23andMe at: 2024-01-01
# rsid	chromosome	position	genotype
rs1234567	1	100	AA
"""
        with tempfile.NamedTemporaryFile(mode='w', suffix='.txt', delete=False) as f:
            f.write(data_23andme)
            f.flush()
            
            try:
                fmt = detect_format(f.name)
                assert fmt == '23andme'
            finally:
                os.unlink(f.name)
    
    def test_detect_ancestry(self):
        """Test AncestryDNA format detection."""
        from comprehensive_analysis import detect_format
        
        data_ancestry = """# AncestryDNA raw data download
rsid	chromosome	position	allele1	allele2
rs1234567	1	100	A	G
"""
        with tempfile.NamedTemporaryFile(mode='w', suffix='.txt', delete=False) as f:
            f.write(data_ancestry)
            f.flush()
            
            try:
                fmt = detect_format(f.name)
                assert fmt == 'ancestry'
            finally:
                os.unlink(f.name)


class TestEdgeCases:
    """Tests for edge cases in DNA data."""
    
    def test_missing_genotypes(self):
        """Test handling of missing genotype data."""
        from comprehensive_analysis import load_consumer_format
        
        with tempfile.NamedTemporaryFile(mode='w', suffix='.txt', delete=False) as f:
            f.write(SYNTHETIC_GENOME_EDGE_CASES)
            f.flush()
            
            try:
                genotypes = load_consumer_format(f.name)
                
                # Should skip invalid entries
                assert 'invalid_rsid' not in genotypes
                
                # Should have valid entries
                assert 'rs1801133' in genotypes
                assert 'rs4680' in genotypes
                
                # Should skip -- and 00 genotypes
                assert genotypes.get('rs6025') is None or genotypes['rs6025'] not in ['--', '00']
            finally:
                os.unlink(f.name)
    
    def test_empty_file(self):
        """Test handling of empty file."""
        from comprehensive_analysis import load_consumer_format
        
        with tempfile.NamedTemporaryFile(mode='w', suffix='.txt', delete=False) as f:
            f.write("# Header only\n")
            f.flush()
            
            try:
                genotypes = load_consumer_format(f.name)
                assert len(genotypes) == 0
            finally:
                os.unlink(f.name)
    
    def test_gzip_vcf(self):
        """Test gzipped VCF handling."""
        import gzip
        from comprehensive_analysis import load_vcf
        
        with tempfile.NamedTemporaryFile(mode='wb', suffix='.vcf.gz', delete=False) as f:
            f.write(gzip.compress(SYNTHETIC_VCF.encode()))
            f.flush()
            
            try:
                genotypes = load_vcf(f.name)
                assert len(genotypes) > 0
            finally:
                os.unlink(f.name)


class TestAPOEDetermination:
    """Tests for APOE genotype determination."""
    
    def test_apoe_e3e4(self):
        """Test APOE ε3/ε4 determination."""
        from comprehensive_analysis import determine_apoe
        
        # E3/E4: rs429358=TC, rs7412=CC
        genotypes = {'rs429358': 'TC', 'rs7412': 'CC'}
        result = determine_apoe(genotypes)
        
        assert result['genotype'] == 'ε3/ε4'
        assert result['risk_level'] == 'elevated'
    
    def test_apoe_e3e3(self):
        """Test APOE ε3/ε3 determination."""
        from comprehensive_analysis import determine_apoe
        
        # E3/E3: rs429358=TT, rs7412=CC
        genotypes = {'rs429358': 'TT', 'rs7412': 'CC'}
        result = determine_apoe(genotypes)
        
        assert result['genotype'] == 'ε3/ε3'
        assert result['risk_level'] == 'average'
    
    def test_apoe_e4e4(self):
        """Test APOE ε4/ε4 determination."""
        from comprehensive_analysis import determine_apoe
        
        # E4/E4: rs429358=CC, rs7412=CC
        genotypes = {'rs429358': 'CC', 'rs7412': 'CC'}
        result = determine_apoe(genotypes)
        
        assert result['genotype'] == 'ε4/ε4'
        assert result['risk_level'] == 'high'
    
    def test_apoe_e2e3(self):
        """Test APOE ε2/ε3 determination."""
        from comprehensive_analysis import determine_apoe
        
        # E2/E3: rs429358=TT, rs7412=TC
        genotypes = {'rs429358': 'TT', 'rs7412': 'TC'}
        result = determine_apoe(genotypes)
        
        assert result['genotype'] in ['ε2/ε3', 'ε3/ε2']
        assert result['risk_level'] == 'low'
    
    def test_apoe_missing_data(self):
        """Test APOE with missing SNPs."""
        from comprehensive_analysis import determine_apoe
        
        genotypes = {'rs429358': 'TT'}  # Missing rs7412
        result = determine_apoe(genotypes)
        
        assert result['genotype'] == 'unknown'


if __name__ == '__main__':
    pytest.main([__file__, '-v'])
