#!/usr/bin/env python3
"""Split PDF into separate files by page or page ranges."""

import sys
import argparse
from pathlib import Path

try:
    from PyPDF2 import PdfReader, PdfWriter
except ImportError:
    print("Error: PyPDF2 not installed. Run: pip3 install PyPDF2", file=sys.stderr)
    sys.exit(1)


def split_pdf(pdf_path, output_dir, mode='pages', ranges=None):
    """
    Split PDF into separate files.
    
    Args:
        pdf_path: Path to input PDF
        output_dir: Directory for output files
        mode: 'pages' (one file per page) or 'ranges' (custom ranges)
        ranges: List of page ranges [(start, end), ...] for mode='ranges' (1-indexed, inclusive)
    """
    pdf_path = Path(pdf_path)
    output_dir = Path(output_dir)
    
    if not pdf_path.exists():
        print(f"Error: File not found: {pdf_path}", file=sys.stderr)
        sys.exit(1)
    
    output_dir.mkdir(parents=True, exist_ok=True)
    
    reader = PdfReader(str(pdf_path))
    total_pages = len(reader.pages)
    
    base_name = pdf_path.stem
    
    if mode == 'pages':
        # Split into individual pages
        for i in range(total_pages):
            writer = PdfWriter()
            writer.add_page(reader.pages[i])
            
            output_path = output_dir / f"{base_name}_page_{i+1}.pdf"
            with open(output_path, 'wb') as f:
                writer.write(f)
            
            print(f"Created: {output_path}")
    
    elif mode == 'ranges' and ranges:
        # Split by custom ranges
        for idx, (start, end) in enumerate(ranges, 1):
            writer = PdfWriter()
            
            # Convert to 0-indexed and clamp to valid range
            start_idx = max(0, start - 1)
            end_idx = min(total_pages, end)
            
            for i in range(start_idx, end_idx):
                writer.add_page(reader.pages[i])
            
            output_path = output_dir / f"{base_name}_part_{idx}_pages_{start}-{end}.pdf"
            with open(output_path, 'wb') as f:
                writer.write(f)
            
            print(f"Created: {output_path} (pages {start}-{end})")
    
    print(f"\nAll files saved to: {output_dir}")


def main():
    parser = argparse.ArgumentParser(description='Split PDF into separate files')
    parser.add_argument('pdf', help='Path to input PDF')
    parser.add_argument('-o', '--output-dir', required=True, help='Output directory')
    parser.add_argument('-m', '--mode', choices=['pages', 'ranges'], default='pages',
                       help='Split mode: pages (one per page) or ranges (custom)')
    parser.add_argument('-r', '--ranges', type=str,
                       help='Page ranges for mode=ranges (e.g., "1-3,5-7,10-12")')
    
    args = parser.parse_args()
    
    ranges = None
    if args.mode == 'ranges':
        if not args.ranges:
            print("Error: --ranges required for mode=ranges", file=sys.stderr)
            sys.exit(1)
        
        # Parse ranges like "1-3,5-7,10-12"
        ranges = []
        for r in args.ranges.split(','):
            start, end = map(int, r.strip().split('-'))
            ranges.append((start, end))
    
    split_pdf(args.pdf, args.output_dir, args.mode, ranges)


if __name__ == '__main__':
    main()
