#!/usr/bin/env python3
"""Rotate pages in a PDF file."""

import sys
import argparse
from pathlib import Path

try:
    from PyPDF2 import PdfReader, PdfWriter
except ImportError:
    print("Error: PyPDF2 not installed. Run: pip3 install PyPDF2", file=sys.stderr)
    sys.exit(1)


def rotate_pdf(pdf_path, output_path, rotation=90, pages=None):
    """
    Rotate PDF pages.
    
    Args:
        pdf_path: Path to input PDF
        output_path: Path to output PDF
        rotation: Rotation angle (90, 180, 270, or -90)
        pages: List of page numbers (1-indexed) or None for all pages
    """
    pdf_path = Path(pdf_path)
    
    if not pdf_path.exists():
        print(f"Error: File not found: {pdf_path}", file=sys.stderr)
        sys.exit(1)
    
    if rotation not in [90, 180, 270, -90]:
        print(f"Error: Invalid rotation angle: {rotation}. Use 90, 180, 270, or -90", file=sys.stderr)
        sys.exit(1)
    
    reader = PdfReader(str(pdf_path))
    writer = PdfWriter()
    
    total_pages = len(reader.pages)
    
    if pages:
        pages_to_rotate = set(p - 1 for p in pages if 0 < p <= total_pages)  # Convert to 0-indexed
    else:
        pages_to_rotate = set(range(total_pages))
    
    for i in range(total_pages):
        page = reader.pages[i]
        
        if i in pages_to_rotate:
            page.rotate(rotation)
        
        writer.add_page(page)
    
    with open(output_path, 'wb') as f:
        writer.write(f)
    
    print(f"Rotated PDF saved to: {output_path}")
    if pages:
        print(f"Rotated pages: {sorted([p+1 for p in pages_to_rotate])}")
    else:
        print("Rotated all pages")


def main():
    parser = argparse.ArgumentParser(description='Rotate pages in a PDF')
    parser.add_argument('pdf', help='Path to input PDF')
    parser.add_argument('-o', '--output', required=True, help='Output PDF file path')
    parser.add_argument('-r', '--rotation', type=int, default=90, 
                       choices=[90, 180, 270, -90],
                       help='Rotation angle (default: 90)')
    parser.add_argument('-p', '--pages', type=int, nargs='+',
                       help='Specific pages to rotate (1-indexed, e.g., -p 1 3 5)')
    
    args = parser.parse_args()
    rotate_pdf(args.pdf, args.output, args.rotation, args.pages)


if __name__ == '__main__':
    main()
