#!/usr/bin/env python3
"""Get metadata and structure information from PDF files."""

import sys
import json
import argparse
from pathlib import Path

try:
    import pdfplumber
except ImportError:
    print("Error: pdfplumber not installed. Run: pip3 install pdfplumber", file=sys.stderr)
    sys.exit(1)


def get_pdf_info(pdf_path, format='text'):
    """
    Get PDF metadata and structure info.
    
    Args:
        pdf_path: Path to PDF file
        format: Output format ('text' or 'json')
    """
    pdf_path = Path(pdf_path)
    
    if not pdf_path.exists():
        print(f"Error: File not found: {pdf_path}", file=sys.stderr)
        sys.exit(1)
    
    with pdfplumber.open(pdf_path) as pdf:
        info = {
            'file': str(pdf_path),
            'pages': len(pdf.pages),
            'metadata': pdf.metadata or {},
        }
        
        # Add page sizes
        page_sizes = []
        for i, page in enumerate(pdf.pages, 1):
            page_sizes.append({
                'page': i,
                'width': page.width,
                'height': page.height,
            })
        info['page_sizes'] = page_sizes
        
        if format == 'json':
            print(json.dumps(info, indent=2, ensure_ascii=False))
        else:
            print(f"File: {info['file']}")
            print(f"Pages: {info['pages']}")
            print("\nMetadata:")
            for key, value in info['metadata'].items():
                print(f"  {key}: {value}")
            print("\nPage Sizes:")
            for ps in page_sizes:
                print(f"  Page {ps['page']}: {ps['width']:.1f} x {ps['height']:.1f} pt")


def main():
    parser = argparse.ArgumentParser(description='Get PDF metadata and structure info')
    parser.add_argument('pdf', help='Path to PDF file')
    parser.add_argument('-f', '--format', choices=['text', 'json'], default='text',
                       help='Output format (default: text)')
    
    args = parser.parse_args()
    get_pdf_info(args.pdf, args.format)


if __name__ == '__main__':
    main()
