#!/usr/bin/env python3
"""Merge multiple PDF files into one."""

import sys
import argparse
from pathlib import Path

try:
    from PyPDF2 import PdfMerger
except ImportError:
    print("Error: PyPDF2 not installed. Run: pip3 install PyPDF2", file=sys.stderr)
    sys.exit(1)


def merge_pdfs(pdf_files, output_path):
    """
    Merge multiple PDFs into one.
    
    Args:
        pdf_files: List of PDF file paths to merge
        output_path: Output PDF file path
    """
    merger = PdfMerger()
    
    for pdf_file in pdf_files:
        pdf_path = Path(pdf_file)
        if not pdf_path.exists():
            print(f"Warning: File not found, skipping: {pdf_file}", file=sys.stderr)
            continue
        
        merger.append(str(pdf_path))
        print(f"Added: {pdf_file}")
    
    output_path = Path(output_path)
    merger.write(str(output_path))
    merger.close()
    
    print(f"\nMerged PDF saved to: {output_path}")


def main():
    parser = argparse.ArgumentParser(description='Merge multiple PDF files')
    parser.add_argument('pdfs', nargs='+', help='PDF files to merge')
    parser.add_argument('-o', '--output', required=True, help='Output PDF file path')
    
    args = parser.parse_args()
    merge_pdfs(args.pdfs, args.output)


if __name__ == '__main__':
    main()
