#!/usr/bin/env python3
"""Extract text from PDF files with page-by-page or full extraction."""

import sys
import argparse
from pathlib import Path

try:
    import pdfplumber
except ImportError:
    print("Error: pdfplumber not installed. Run: pip3 install pdfplumber", file=sys.stderr)
    sys.exit(1)


def extract_text(pdf_path, pages=None, output=None):
    """
    Extract text from PDF.
    
    Args:
        pdf_path: Path to PDF file
        pages: List of page numbers (1-indexed) or None for all pages
        output: Output file path or None for stdout
    """
    pdf_path = Path(pdf_path)
    
    if not pdf_path.exists():
        print(f"Error: File not found: {pdf_path}", file=sys.stderr)
        sys.exit(1)
    
    result = []
    
    with pdfplumber.open(pdf_path) as pdf:
        total_pages = len(pdf.pages)
        
        if pages:
            page_nums = [p - 1 for p in pages if 0 < p <= total_pages]  # Convert to 0-indexed
        else:
            page_nums = range(total_pages)
        
        for page_num in page_nums:
            page = pdf.pages[page_num]
            text = page.extract_text()
            
            if text:
                result.append(f"=== Page {page_num + 1} ===\n{text}\n")
    
    output_text = "\n".join(result)
    
    if output:
        Path(output).write_text(output_text, encoding='utf-8')
        print(f"Text extracted to: {output}")
    else:
        print(output_text)


def main():
    parser = argparse.ArgumentParser(description='Extract text from PDF files')
    parser.add_argument('pdf', help='Path to PDF file')
    parser.add_argument('-p', '--pages', type=int, nargs='+', 
                       help='Specific pages to extract (1-indexed, e.g., -p 1 3 5)')
    parser.add_argument('-o', '--output', help='Output file path (default: stdout)')
    
    args = parser.parse_args()
    
    extract_text(args.pdf, args.pages, args.output)


if __name__ == '__main__':
    main()
