#!/usr/bin/env python3
"""Edit text in PDF files by replacing content."""

import sys
import argparse
from pathlib import Path

try:
    from PyPDF2 import PdfReader, PdfWriter
    from reportlab.pdfgen import canvas
    from reportlab.lib.pagesizes import letter
    from io import BytesIO
except ImportError as e:
    print(f"Error: Missing dependency. Run: pip3 install PyPDF2 reportlab", file=sys.stderr)
    sys.exit(1)


def overlay_text(pdf_path, output_path, page_num, text, x, y, font_size=12):
    """
    Add/overlay text on a PDF page.
    
    Args:
        pdf_path: Path to input PDF
        output_path: Path to output PDF
        page_num: Page number (1-indexed)
        text: Text to add
        x: X coordinate (from left)
        y: Y coordinate (from bottom)
        font_size: Font size
    """
    pdf_path = Path(pdf_path)
    
    if not pdf_path.exists():
        print(f"Error: File not found: {pdf_path}", file=sys.stderr)
        sys.exit(1)
    
    reader = PdfReader(str(pdf_path))
    writer = PdfWriter()
    
    total_pages = len(reader.pages)
    
    if not (1 <= page_num <= total_pages):
        print(f"Error: Invalid page number {page_num}. PDF has {total_pages} pages.", file=sys.stderr)
        sys.exit(1)
    
    # Create overlay with text
    packet = BytesIO()
    can = canvas.Canvas(packet, pagesize=letter)
    can.setFont("Helvetica", font_size)
    can.drawString(x, y, text)
    can.save()
    
    packet.seek(0)
    overlay_pdf = PdfReader(packet)
    
    # Merge pages
    for i in range(total_pages):
        page = reader.pages[i]
        
        # Add overlay to target page
        if i == page_num - 1:
            page.merge_page(overlay_pdf.pages[0])
        
        writer.add_page(page)
    
    with open(output_path, 'wb') as f:
        writer.write(f)
    
    print(f"Text added to page {page_num}")
    print(f"Output saved to: {output_path}")


def replace_text_simple(pdf_path, output_path, find_text, replace_text):
    """
    Simple text replacement in PDF metadata and content streams.
    Note: This is basic and may not work for all PDFs due to PDF complexity.
    
    Args:
        pdf_path: Path to input PDF
        output_path: Path to output PDF
        find_text: Text to find
        replace_text: Text to replace with
    """
    pdf_path = Path(pdf_path)
    
    if not pdf_path.exists():
        print(f"Error: File not found: {pdf_path}", file=sys.stderr)
        sys.exit(1)
    
    print("⚠️  Warning: PDF text replacement is limited due to PDF format complexity.")
    print("   This works best for simple text in metadata.")
    print("   For complex editing, consider extracting text, editing, and regenerating PDF.\n")
    
    # Read PDF
    with open(pdf_path, 'rb') as f:
        pdf_data = f.read()
    
    # Simple byte-level replacement (limited)
    find_bytes = find_text.encode('utf-8')
    replace_bytes = replace_text.encode('utf-8')
    
    modified_data = pdf_data.replace(find_bytes, replace_bytes)
    
    # Write output
    with open(output_path, 'wb') as f:
        f.write(modified_data)
    
    if pdf_data != modified_data:
        print(f"✓ Text replacement attempted")
        print(f"  Found and replaced: '{find_text}' → '{replace_text}'")
    else:
        print(f"✗ Text not found: '{find_text}'")
    
    print(f"Output saved to: {output_path}")


def main():
    parser = argparse.ArgumentParser(
        description='Edit text in PDF files',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Add text overlay on page 1
  %(prog)s input.pdf -o output.pdf --overlay "New Text" --page 1 --x 100 --y 700
  
  # Replace text (limited)
  %(prog)s input.pdf -o output.pdf --replace "Old" "New"
        """
    )
    
    parser.add_argument('pdf', help='Path to input PDF')
    parser.add_argument('-o', '--output', required=True, help='Output PDF file path')
    
    # Overlay mode
    parser.add_argument('--overlay', type=str, help='Text to overlay on page')
    parser.add_argument('--page', type=int, help='Page number for overlay (1-indexed)')
    parser.add_argument('--x', type=float, default=100, help='X coordinate (default: 100)')
    parser.add_argument('--y', type=float, default=700, help='Y coordinate (default: 700)')
    parser.add_argument('--font-size', type=int, default=12, help='Font size (default: 12)')
    
    # Replace mode
    parser.add_argument('--replace', nargs=2, metavar=('FIND', 'REPLACE'),
                       help='Replace text: --replace "old" "new"')
    
    args = parser.parse_args()
    
    # Validate mode selection
    if args.overlay and args.replace:
        print("Error: Cannot use --overlay and --replace together. Choose one.", file=sys.stderr)
        sys.exit(1)
    
    if not args.overlay and not args.replace:
        print("Error: Must specify either --overlay or --replace", file=sys.stderr)
        parser.print_help()
        sys.exit(1)
    
    # Overlay mode
    if args.overlay:
        if not args.page:
            print("Error: --page required for overlay mode", file=sys.stderr)
            sys.exit(1)
        
        overlay_text(args.pdf, args.output, args.page, args.overlay, 
                    args.x, args.y, args.font_size)
    
    # Replace mode
    elif args.replace:
        find_text, replace_text = args.replace
        replace_text_simple(args.pdf, args.output, find_text, replace_text)


if __name__ == '__main__':
    main()
