---
name: PDF Generator
slug: pdf-generator
version: 1.0.1
homepage: https://clawic.com/skills/pdf-generator
description: Generate professional PDFs from Markdown, HTML, data, or code. Reports, invoices, contracts, and documents with best practices.
metadata: {"clawdbot":{"emoji":"📄","requires":{"bins":[]},"os":["linux","darwin","win32"]}}
---

## When to Use

User needs to create, generate, or export PDF documents. Agent handles document generation from multiple sources (Markdown, HTML, JSON, templates), formatting, styling, and batch processing.

## Scope

This skill ONLY:
- Provides code patterns and implementation guidance for PDF generation
- Explains tool selection, CSS for print, and document structure
- Shows reference examples for common document types

This skill NEVER:
- Executes code or generates files directly
- Makes network requests
- Accesses files outside user's working directory

All code examples are reference patterns for the user to implement.

## Quick Reference

| Topic | File |
|-------|------|
| Tool selection | `tools.md` |
| Document types | `templates.md` |
| Advanced operations | `advanced.md` |

## Core Rules

### 1. Choose the Right Tool

| Source | Best Tool | Why |
|--------|-----------|-----|
| Markdown | pandoc | Native support, TOC, templates |
| HTML/CSS | weasyprint | Best CSS support, no LaTeX |
| Data/JSON | reportlab | Programmatic, precise control |
| Simple text | fpdf2 | Lightweight, fast |

**Default recommendation:** weasyprint for most HTML-based documents.

### 2. Structure Before Style

```python
# CORRECT: semantic structure
html = """
<article>
  <header><h1>Report Title</h1></header>
  <section>
    <h2>Summary</h2>
    <p>Content...</p>
  </section>
</article>
"""

# WRONG: style-first approach
html = "<div style='font-size:24px'>Report Title</div>"
```

### 3. Handle Page Breaks Explicitly

```css
/* Force page break before */
.new-page { page-break-before: always; }

/* Keep together */
.keep-together { page-break-inside: avoid; }

/* Headers never orphaned */
h2, h3 { page-break-after: avoid; }
```

### 4. Always Set Metadata

```python
# Example pattern for weasyprint
html = """
<html>
<head>
  <title>Document Title</title>
  <meta name="author" content="Author Name">
</head>
...
"""
```

### 5. Use Print-Optimized CSS

```css
@media print {
  body {
    font-family: 'Georgia', serif;
    font-size: 11pt;
    line-height: 1.5;
  }
  
  @page {
    size: A4;
    margin: 2cm;
  }
  
  .no-print { display: none; }
}
```

### 6. Validate Output

After generating any PDF:
1. Check file size (0 bytes = failed)
2. Open and verify page count
3. Verify fonts render correctly

## Common Traps

| Trap | Consequence | Fix |
|------|-------------|-----|
| Missing fonts | Fallback to defaults | Use web-safe fonts |
| Absolute image paths | Images missing | Use relative paths |
| No page size | Unpredictable layout | Set `@page { size: A4; }` |
| Large images | Huge files | Compress before use |

## Security & Privacy

**This is a reference skill.** It provides patterns and guidance only.

**Data that stays local:**
- All PDF generation happens on user's machine
- No data sent externally

**This skill does NOT:**
- Execute code or make files
- Make network requests
- Access system files

## Feedback

- If useful: `clawhub star pdf-generator`
- Stay updated: `clawhub sync`
